"""
Authentication Service
Central authentication and authorization functions
"""

import bcrypt
import jwt
import logging
from datetime import datetime, timezone, timedelta
from fastapi import HTTPException, Depends
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
from database import get_db
import os

# Security
security = HTTPBearer()
JWT_SECRET = os.environ.get('JWT_SECRET', 'mlm-secret-key-change-in-production')
JWT_ALGORITHM = 'HS256'


def hash_password(password: str) -> str:
    """Hash a password using bcrypt"""
    return bcrypt.hashpw(password.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')


def verify_password(plain_password: str, hashed_password: str) -> bool:
    """Verify a password against its hash"""
    try:
        return bcrypt.checkpw(plain_password.encode('utf-8'), hashed_password.encode('utf-8'))
    except Exception:
        return False


def create_access_token(data: dict) -> str:
    """Create a JWT access token"""
    to_encode = data.copy()
    expire = datetime.now(timezone.utc) + timedelta(days=30)
    to_encode.update({"exp": expire})
    return jwt.encode(to_encode, JWT_SECRET, algorithm=JWT_ALGORITHM)


def decode_token(token: str) -> dict:
    """Decode and validate a JWT token"""
    try:
        payload = jwt.decode(token, JWT_SECRET, algorithms=[JWT_ALGORITHM])
        return payload
    except jwt.ExpiredSignatureError:
        raise HTTPException(status_code=401, detail="توکن منقضی شده است")
    except jwt.InvalidTokenError:
        raise HTTPException(status_code=401, detail="توکن نامعتبر است")


async def get_current_user(credentials: HTTPAuthorizationCredentials = Depends(security)):
    """Get current user from JWT token"""
    payload = decode_token(credentials.credentials)
    
    db = await get_db()
    user = await db.users.find_one({"user_id": payload.get("user_id")}, {"_id": 0})
    if not user:
        raise HTTPException(status_code=401, detail="کاربر یافت نشد")
    
    # Check if account is banned
    if user.get("status") == "banned":
        raise HTTPException(status_code=403, detail="حساب کاربری شما مسدود شده است")
    
    return user


async def get_admin_user(current_user: dict = Depends(get_current_user)):
    """Verify admin access"""
    if current_user.get("role") not in ["admin", "seed"]:
        logging.warning(f"Unauthorized admin access attempt by user {current_user.get('user_id')}")
        raise HTTPException(status_code=403, detail="دسترسی محدود به ادمین")
    return current_user
