"""
Database Connection and Utilities
مدیریت اتصال به MongoDB
"""
from motor.motor_asyncio import AsyncIOMotorClient
from config import MONGO_URL, DB_NAME
import logging

logger = logging.getLogger(__name__)

# Global database connection
client = None
db = None

def get_database():
    """برگرداندن instance دیتابیس"""
    global db
    if db is None:
        raise RuntimeError("Database not initialized. Call init_database() first.")
    return db


async def get_db():
    """
    تابع async برای دریافت دیتابیس
    این تابع برای استفاده در Depends() در FastAPI
    """
    return get_database()


async def init_database():
    """مقداردهی اولیه اتصال به دیتابیس"""
    global client, db
    
    try:
        client = AsyncIOMotorClient(MONGO_URL)
        db = client[DB_NAME]
        
        # Test connection
        await db.command('ping')
        logger.info(f"✅ Connected to MongoDB: {DB_NAME}")
        
        # Initialize seed user and settings
        await init_seed_user()
        
        return db
    except Exception as e:
        logger.error(f"❌ Failed to connect to MongoDB: {e}")
        raise

async def close_database():
    """بستن اتصال دیتابیس"""
    global client
    if client:
        client.close()
        logger.info("🔒 Database connection closed")

async def init_seed_user():
    """ایجاد کاربر SEED اگر وجود نداشته باشد"""
    from datetime import datetime, timezone
    import bcrypt
    
    seed_user = await db.users.find_one({"user_id": "seed"})
    if not seed_user:
        hashed_password = bcrypt.hashpw("seed123".encode('utf-8'), bcrypt.gensalt())
        seed_user = {
            "user_id": "seed",
            "email": "seed@system.local",
            "password": hashed_password.decode('utf-8'),
            "role": "seed",
            "status": "active",
            "depth": 0,
            "direct_children": [],
            "referrer_id": None,
            "placement_parent_id": None,
            "total_points": 0,
            "is_tree_complete": False,
            "blocked_level_reward": False,
            "created_at": datetime.now(timezone.utc).isoformat(),
            "referral_code": "SEED"
        }
        await db.users.insert_one(seed_user)
        logger.info("✅ Seed user created")
        
        # ثبت پرداخت اولیه seed
        seed_payment = await db.payments.find_one({"user_id": "seed"})
        if not seed_payment:
            settings_data = await db.settings.find_one({"setting_id": "rewards"})
            seed_initial = settings_data.get("seed_initial_payment", 100.0) if settings_data else 100.0
            
            seed_payment_record = {
                "payment_id": "seed-initial",
                "user_id": "seed",
                "amount": seed_initial,
                "timestamp": datetime.now(timezone.utc).isoformat(),
                "description": "سرمایه اولیه سیستم (seed)"
            }
            await db.payments.insert_one(seed_payment_record)
            logger.info("✅ Seed initial payment created")

# Export db for backward compatibility
__all__ = ['db', 'get_db', 'get_database', 'init_database', 'close_database']
