import { useState, useEffect } from 'react';
import axios from 'axios';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { ExternalLink, AlertCircle, ArrowLeft } from 'lucide-react';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

export default function IframePage({ configKey, title, icon, description }) {
  const [url, setUrl] = useState('');
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchUrl();
  }, []);

  const fetchUrl = async () => {
    try {
      const res = await axios.get(`${API}/user/config/public`);
      setUrl(res.data[configKey] || '');
    } catch (error) {
      console.error('Error fetching URL:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleOpenPage = () => {
    if (url) {
      window.open(url, '_blank', 'noopener,noreferrer');
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-96">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  if (!url) {
    return (
      <Card className="shadow-lg border-0">
        <CardContent className="p-12 text-center">
          <AlertCircle className="w-16 h-16 mx-auto text-gray-400 mb-4" />
          <h3 className="text-xl font-semibold text-gray-700 mb-2">محتوایی یافت نشد</h3>
          <p className="text-gray-500">این بخش هنوز توسط مدیر تنظیم نشده است</p>
        </CardContent>
      </Card>
    );
  }

  return (
    <div className="max-w-2xl mx-auto">
      <Card className="shadow-xl border-0 bg-gradient-to-br from-white to-gray-50">
        <CardHeader className="text-center space-y-4 pb-8">
          <div className="mx-auto w-20 h-20 bg-gradient-to-br from-blue-500 to-purple-600 rounded-full flex items-center justify-center shadow-lg">
            {icon && <div className="text-white scale-150">{icon}</div>}
            {!icon && <ExternalLink className="w-10 h-10 text-white" />}
          </div>
          <CardTitle className="text-3xl font-bold bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
            {title}
          </CardTitle>
          {description && (
            <CardDescription className="text-base text-gray-600">
              {description}
            </CardDescription>
          )}
        </CardHeader>
        
        <CardContent className="space-y-6 pb-10">
          <div className="bg-blue-50 border-r-4 border-blue-500 p-6 rounded-lg">
            <p className="text-gray-700 text-center leading-relaxed">
              برای مشاهده محتوای این بخش، روی دکمه زیر کلیک کنید.
              <br />
              صفحه در یک پنجره جدید باز خواهد شد.
            </p>
          </div>

          <div className="flex justify-center pt-4">
            <Button
              onClick={handleOpenPage}
              size="lg"
              className="bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white font-semibold px-8 py-6 text-lg shadow-lg hover:shadow-xl transition-all duration-200 transform hover:scale-105"
            >
              <ExternalLink className="w-5 h-5 ml-2" />
              {title === 'صفحه توضیحات' && 'ورود به صفحه توضیحات'}
              {title === 'صفحه آموزش' && 'ورود به صفحه آموزش'}
              {!['صفحه توضیحات', 'صفحه آموزش'].includes(title) && `ورود به ${title}`}
              <ArrowLeft className="w-5 h-5 mr-2" />
            </Button>
          </div>

          <div className="text-center pt-2">
            <a
              href={url}
              target="_blank"
              rel="noopener noreferrer"
              className="text-sm text-gray-500 hover:text-gray-700 underline decoration-dotted"
            >
              {url.length > 50 ? url.substring(0, 50) + '...' : url}
            </a>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
