"""Reward Distributor - توزیع پاداش با bulk operations"""

from typing import Dict, List
import uuid
import logging
from datetime import datetime, timezone
from .calculator import RewardCalculator

logger = logging.getLogger(__name__)

_db = None

def set_database(db):
    """تنظیم database connection"""
    global _db
    _db = db


class RewardDistributor:
    """توزیع‌کننده پاداش - با bulk operations"""
    
    @staticmethod
    async def distribute_all_rewards(new_user_id: str):
        """
        توزیع کامل پاداش‌ها - real-time و بدون traversal
        
        Process:
        1. Get settings (1 query)
        2. Get new user with ancestor_depths (1 query)
        3. Check duplicate lock (1 query)
        4. Calculate direct reward (2 queries)
        5. Calculate level rewards (1 query)
        6. Bulk update points (1 query)
        7. Bulk insert transactions (1 query)
        
        Total Queries: 8
        قبلی: 20-40 queries!
        
        Args:
            new_user_id: شناسه کاربر جدید فعال شده
        """
        
        # Query 1: Get settings
        settings = await _db.settings.find_one({"setting_id": "rewards"})
        if not settings:
            logger.error("❌ Reward settings not found!")
            return
        
        # Query 2: Get new user (با ancestor_depths)
        new_user = await _db.users.find_one(
            {"user_id": new_user_id},
            {"_id": 0, "user_id": 1, "referrer_id": 1, "status": 1, 
             "depth": 1, "ancestor_depths": 1}
        )
        
        if not new_user or new_user.get("status") != "active":
            logger.warning(f"⚠️ User {new_user_id} not active or not found")
            return
        
        # Query 3: Check duplicate lock
        lock_id = f"reward_lock_{new_user_id}_direct"
        lock_result = await _db.reward_locks.update_one(
            {"lock_id": lock_id},
            {"$setOnInsert": {
                "lock_id": lock_id,
                "new_user_id": new_user_id,
                "created_at": datetime.now(timezone.utc).isoformat()
            }},
            upsert=True
        )
        
        if lock_result.upserted_id is None and lock_result.modified_count == 0:
            logger.info(f"⚠️ Rewards already processed for {new_user_id}")
            return
        
        # محاسبه همه پاداش‌ها
        rewards = []
        
        # Direct reward (Queries 4-5: inside calculate_direct_reward)
        referrer_id = new_user.get("referrer_id")
        if referrer_id:
            direct_reward = await RewardCalculator.calculate_direct_reward(
                referrer_id,
                new_user_id,
                settings
            )
            if direct_reward:
                rewards.append(direct_reward)
        
        # Level rewards (Query 6: inside calculate_level_rewards)
        level_rewards = await RewardCalculator.calculate_level_rewards(
            new_user,
            settings
        )
        rewards.extend(level_rewards)
        
        if not rewards:
            logger.info(f"ℹ️ No rewards to distribute for {new_user_id}")
            return
        
        # Prepare bulk operations
        from pymongo import UpdateOne
        
        updates = []
        transactions = []
        
        for reward in rewards:
            # Update points
            updates.append(
                UpdateOne(
                    {"user_id": reward["recipient_id"]},
                    {"$inc": {"total_points": reward["amount"]}}
                )
            )
            
            # Transaction record
            transactions.append({
                "transaction_id": str(uuid.uuid4()),
                "user_id": reward["recipient_id"],
                "from_user_id": new_user_id,
                "reward_type": reward["reward_type"],
                "amount": reward["amount"],
                "timestamp": datetime.now(timezone.utc).isoformat(),
                "description": reward["description"]
            })
        
        # Query 7: Bulk update points
        if updates:
            result = await _db.users.bulk_write(updates)
            logger.info(f"✅ Updated {result.modified_count} users with rewards")
        
        # Query 8: Bulk insert transactions
        if transactions:
            await _db.transactions.insert_many(transactions)
            logger.info(f"✅ Inserted {len(transactions)} transactions")
        
        # Log summary
        total_amount = sum(r["amount"] for r in rewards)
        logger.info(f"🎉 Distributed ${total_amount} to {len(rewards)} recipients for user {new_user_id}")
