#!/bin/bash

################################################################################
# MLM System - نصب خودکار کامل با Production Setup
# Compatible: Ubuntu 18+, Debian 10+, CentOS 7+, Rocky Linux
# Auto-installs: تمام وابستگی‌ها + MongoDB + Redis + PM2 + Nginx
# Version: 4.0 Production Ready
################################################################################

set -e
set -o pipefail

# رنگ‌ها
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
BOLD='\033[1m'
NC='\033[0m'

# تنظیمات
MONGO_PORT=${MONGO_PORT:-27017}
REDIS_PORT=${REDIS_PORT:-6379}
BACKEND_PORT=${BACKEND_PORT:-8001}
FRONTEND_PORT=${FRONTEND_PORT:-3000}
NGINX_PORT=${NGINX_PORT:-80}
DB_NAME=${DB_NAME:-mlm_production}
INSTALL_DIR=$(pwd)
LOG_FILE="$INSTALL_DIR/install.log"

# تولید JWT secret امن
JWT_SECRET=$(openssl rand -hex 32 2>/dev/null || echo "mlm-secret-$(date +%s)-$(cat /dev/urandom | tr -dc 'a-zA-Z0-9' | fold -w 32 | head -n 1)")

# Detect running as root
if [ "$EUID" -eq 0 ]; then
    SUDO_CMD=""
else
    SUDO_CMD="sudo"
fi

# Logging
exec 1> >(tee -a "$LOG_FILE")
exec 2>&1

# Header
clear
cat << "EOF"
════════════════════════════════════════════════════════════════════════
   🚀 MLM System - نصب خودکار Production-Ready
════════════════════════════════════════════════════════════════════════
EOF
echo ""
echo -e "${BLUE}📍 نسخه: 4.0 Production - Zero Manual Steps${NC}"
echo -e "${BLUE}📍 تاریخ: $(date '+%Y-%m-%d %H:%M:%S')${NC}"
echo ""

# تشخیص IP سرور
SERVER_IP=$(curl -s ifconfig.me 2>/dev/null || curl -s api.ipify.org 2>/dev/null || hostname -I | awk '{print $1}')
echo -e "${GREEN}✔ IP سرور: $SERVER_IP${NC}"
echo ""
sleep 1

# تشخیص OS
if [ -f /etc/os-release ]; then
    . /etc/os-release
    OS_NAME=$ID
    OS_PRETTY=$PRETTY_NAME
else
    echo -e "${RED}❌ نمی‌توانم سیستم‌عامل را تشخیص دهم${NC}"
    exit 1
fi

echo -e "${GREEN}✔ سیستم‌عامل: $OS_PRETTY${NC}"
echo -e "${GREEN}✔ دسترسی: $([ "$EUID" -eq 0 ] && echo 'Root' || echo 'Sudo User')${NC}"
echo ""

################################################################################
# توابع کمکی
################################################################################

log_step() {
    echo ""
    echo "════════════════════════════════════════════════════════════════════════"
    echo -e "${YELLOW}$1${NC}"
    echo "════════════════════════════════════════════════════════════════════════"
    echo ""
}

check_command() {
    command -v "$1" >/dev/null 2>&1
}

check_port() {
    local port=$1
    if $SUDO_CMD netstat -tuln 2>/dev/null | grep -q ":$port " || \
       $SUDO_CMD ss -tuln 2>/dev/null | grep -q ":$port " || \
       $SUDO_CMD lsof -Pi :$port -sTCP:LISTEN -t >/dev/null 2>&1; then
        return 1
    else
        return 0
    fi
}

################################################################################
# 1. پاکسازی نصب‌های قبلی
################################################################################

log_step "🧹 پاکسازی نصب‌های قبلی..."

# توقف PM2 processes قدیمی
echo -e "${YELLOW}⏳ توقف سرویس‌های قدیمی...${NC}"
if check_command pm2; then
    pm2 delete all 2>/dev/null || true
    pm2 save --force 2>/dev/null || true
fi

# توقف systemd services قدیمی
$SUDO_CMD systemctl stop mlm-backend 2>/dev/null || true
$SUDO_CMD systemctl stop mlm-frontend 2>/dev/null || true
$SUDO_CMD systemctl disable mlm-backend 2>/dev/null || true
$SUDO_CMD systemctl disable mlm-frontend 2>/dev/null || true

# حذف containers قدیمی
echo -e "${YELLOW}⏳ حذف containers قدیمی MLM...${NC}"
$SUDO_CMD docker rm -f mlm-mongodb mlm-redis mlm-backend mlm-frontend 2>/dev/null || true
echo -e "${GREEN}✔ Containers قدیمی پاک شد${NC}"

# حذف virtual environments قدیمی
if [ -d "backend/venv" ]; then
    echo -e "${YELLOW}⏳ حذف virtual environment قدیمی...${NC}"
    rm -rf backend/venv
    echo -e "${GREEN}✔ Virtual environment قدیمی پاک شد${NC}"
fi

# حذف node_modules قدیمی
if [ -d "frontend/node_modules" ]; then
    echo -e "${YELLOW}⏳ حذف node_modules قدیمی...${NC}"
    rm -rf frontend/node_modules
    echo -e "${GREEN}✔ node_modules قدیمی پاک شد${NC}"
fi

# حذف build قدیمی
if [ -d "frontend/build" ]; then
    rm -rf frontend/build
fi

echo -e "${GREEN}✔ پاکسازی کامل شد${NC}"

################################################################################
# 2. نصب وابستگی‌های سیستم
################################################################################

log_step "📦 نصب وابستگی‌های سیستم..."

case $OS_NAME in
    ubuntu|debian)
        echo -e "${YELLOW}⏳ به‌روزرسانی package lists...${NC}"
        $SUDO_CMD apt-get update -qq
        
        echo -e "${YELLOW}⏳ نصب ابزارهای پایه...${NC}"
        $SUDO_CMD apt-get install -y \
            curl wget git \
            ca-certificates gnupg lsb-release \
            apt-transport-https software-properties-common \
            openssl net-tools \
            build-essential \
            python3 python3-venv python3-pip python3-dev \
            nginx \
            > /dev/null 2>&1
        ;;
        
    centos|rhel|rocky)
        echo -e "${YELLOW}⏳ نصب ابزارهای پایه...${NC}"
        $SUDO_CMD yum install -y \
            curl wget git \
            ca-certificates gnupg2 \
            openssl net-tools \
            gcc gcc-c++ make \
            python3 python3-pip python3-devel \
            yum-utils \
            nginx \
            > /dev/null 2>&1
        ;;
esac

echo -e "${GREEN}✔ وابستگی‌های سیستم نصب شد${NC}"

################################################################################
# 3. نصب Docker
################################################################################

log_step "🐳 نصب Docker..."

if check_command docker; then
    echo -e "${GREEN}✔ Docker موجود است: $(docker --version | cut -d ' ' -f 3 | cut -d ',' -f 1)${NC}"
else
    echo -e "${YELLOW}⏳ نصب Docker...${NC}"
    
    case $OS_NAME in
        ubuntu|debian)
            $SUDO_CMD mkdir -p /etc/apt/keyrings
            curl -fsSL https://download.docker.com/linux/$OS_NAME/gpg | $SUDO_CMD gpg --dearmor -o /etc/apt/keyrings/docker.gpg
            echo "deb [arch=$(dpkg --print-architecture) signed-by=/etc/apt/keyrings/docker.gpg] https://download.docker.com/linux/$OS_NAME $(lsb_release -cs) stable" | $SUDO_CMD tee /etc/apt/sources.list.d/docker.list > /dev/null
            $SUDO_CMD apt-get update -qq
            $SUDO_CMD apt-get install -y docker-ce docker-ce-cli containerd.io docker-compose-plugin
            ;;
            
        centos|rhel|rocky)
            $SUDO_CMD yum-config-manager --add-repo https://download.docker.com/linux/centos/docker-ce.repo
            $SUDO_CMD yum install -y docker-ce docker-ce-cli containerd.io docker-compose-plugin
            ;;
    esac
    
    $SUDO_CMD systemctl start docker
    $SUDO_CMD systemctl enable docker
    echo -e "${GREEN}✔ Docker نصب و فعال شد${NC}"
fi

# Add user to docker group
if [ "$EUID" -ne 0 ]; then
    $SUDO_CMD usermod -aG docker $USER 2>/dev/null || true
fi

################################################################################
# 4. نصب Node.js و Yarn
################################################################################

log_step "📗 نصب Node.js و Yarn..."

if check_command node; then
    NODE_VER=$(node --version | cut -d 'v' -f 2 | cut -d '.' -f 1)
    if [ "$NODE_VER" -ge 16 ]; then
        echo -e "${GREEN}✔ Node.js موجود است: $(node --version)${NC}"
    else
        INSTALL_NODE=true
    fi
else
    INSTALL_NODE=true
fi

if [ "$INSTALL_NODE" = true ]; then
    echo -e "${YELLOW}⏳ نصب Node.js 18...${NC}"
    
    case $OS_NAME in
        ubuntu|debian)
            curl -fsSL https://deb.nodesource.com/setup_18.x | $SUDO_CMD -E bash -
            $SUDO_CMD apt-get install -y nodejs
            ;;
        centos|rhel|rocky)
            curl -fsSL https://rpm.nodesource.com/setup_18.x | $SUDO_CMD bash -
            $SUDO_CMD yum install -y nodejs
            ;;
    esac
    
    echo -e "${GREEN}✔ Node.js نصب شد: $(node --version)${NC}"
fi

# نصب Yarn
if ! check_command yarn; then
    echo -e "${YELLOW}⏳ نصب Yarn...${NC}"
    $SUDO_CMD npm install -g yarn
fi
echo -e "${GREEN}✔ Yarn: $(yarn --version)${NC}"

################################################################################
# 5. نصب PM2
################################################################################

log_step "⚡ نصب PM2 Process Manager..."

if ! check_command pm2; then
    echo -e "${YELLOW}⏳ نصب PM2...${NC}"
    $SUDO_CMD npm install -g pm2
    
    # Enable PM2 startup
    $SUDO_CMD pm2 startup systemd -u $USER --hp $HOME || true
    
    echo -e "${GREEN}✔ PM2 نصب شد${NC}"
else
    echo -e "${GREEN}✔ PM2 موجود است: $(pm2 --version)${NC}"
fi

################################################################################
# 6. راه‌اندازی MongoDB
################################################################################

log_step "🍃 راه‌اندازی MongoDB..."

# حذف container قدیمی
$SUDO_CMD docker rm -f mlm-mongodb 2>/dev/null || true

# چک کردن port
if ! check_port $MONGO_PORT; then
    echo -e "${RED}❌ Port $MONGO_PORT در حال استفاده است${NC}"
    echo -e "${YELLOW}💡 MongoDB/Service دیگری در حال اجرا است${NC}"
    
    # تلاش برای stop کردن
    $SUDO_CMD systemctl stop mongodb 2>/dev/null || true
    $SUDO_CMD systemctl stop mongod 2>/dev/null || true
    sleep 2
    
    if ! check_port $MONGO_PORT; then
        echo -e "${RED}❌ نمی‌توانم port را آزاد کنم. لطفاً دستی متوقف کنید:${NC}"
        echo "   sudo systemctl stop mongodb"
        exit 1
    fi
fi

echo -e "${YELLOW}⏳ ایجاد MongoDB container...${NC}"

$SUDO_CMD docker run -d \
    --name mlm-mongodb \
    --restart unless-stopped \
    -p 127.0.0.1:$MONGO_PORT:27017 \
    -v mlm-mongodb-data:/data/db \
    -e MONGO_INITDB_DATABASE=$DB_NAME \
    mongo:7.0 \
    > /dev/null

echo -e "${YELLOW}⏳ در انتظار آماده شدن MongoDB...${NC}"
for i in {1..30}; do
    if $SUDO_CMD docker exec mlm-mongodb mongosh --quiet --eval "db.version()" > /dev/null 2>&1; then
        echo -e "${GREEN}✔ MongoDB آماده است${NC}"
        break
    fi
    sleep 1
done

if ! $SUDO_CMD docker exec mlm-mongodb mongosh --quiet --eval "db.version()" > /dev/null 2>&1; then
    echo -e "${RED}❌ MongoDB راه‌اندازی نشد${NC}"
    $SUDO_CMD docker logs mlm-mongodb
    exit 1
fi

################################################################################
# 7. راه‌اندازی Redis
################################################################################

log_step "🔴 راه‌اندازی Redis..."

$SUDO_CMD docker rm -f mlm-redis 2>/dev/null || true

if check_port $REDIS_PORT; then
    echo -e "${YELLOW}⏳ ایجاد Redis container...${NC}"
    
    $SUDO_CMD docker run -d \
        --name mlm-redis \
        --restart unless-stopped \
        -p 127.0.0.1:$REDIS_PORT:6379 \
        -v mlm-redis-data:/data \
        redis:7-alpine \
        redis-server --appendonly yes \
        > /dev/null
    
    sleep 3
    
    if $SUDO_CMD docker exec mlm-redis redis-cli ping 2>/dev/null | grep -q "PONG"; then
        echo -e "${GREEN}✔ Redis راه‌اندازی شد${NC}"
        REDIS_ENABLED=true
    else
        echo -e "${YELLOW}⚠️  Redis اختیاری - ادامه بدون cache${NC}"
        REDIS_ENABLED=false
    fi
else
    echo -e "${YELLOW}⚠️  Port $REDIS_PORT در حال استفاده - Redis نصب نمی‌شود${NC}"
    REDIS_ENABLED=false
fi

################################################################################
# 8. نصب Backend
################################################################################

log_step "🔧 نصب Backend..."

cd "$INSTALL_DIR/backend"

# پاکسازی
rm -rf venv __pycache__ *.pyc 2>/dev/null || true

# ایجاد virtual environment
echo -e "${YELLOW}⏳ ایجاد Python virtual environment...${NC}"
python3 -m venv venv

if [ ! -f "venv/bin/activate" ]; then
    echo -e "${RED}❌ Virtual environment ایجاد نشد${NC}"
    exit 1
fi

# نصب dependencies
echo -e "${YELLOW}⏳ نصب Python packages...${NC}"
source venv/bin/activate

pip install --upgrade pip setuptools wheel > /dev/null 2>&1

if ! pip install -r requirements.txt; then
    echo -e "${RED}❌ خطا در نصب Python packages${NC}"
    cat "$LOG_FILE" | tail -20
    exit 1
fi

deactivate
echo -e "${GREEN}✔ Backend dependencies نصب شد${NC}"

# ایجاد .env امن
echo -e "${YELLOW}⏳ ایجاد فایل .env امن...${NC}"

cat > .env << EOF
# Database Configuration
MONGO_URL=mongodb://localhost:$MONGO_PORT
DB_NAME=$DB_NAME

# Security (Generated)
JWT_SECRET=$JWT_SECRET

# Server Configuration
HOST=0.0.0.0
PORT=$BACKEND_PORT

# CORS - دسترسی از همه جا
CORS_ORIGINS=http://localhost:$NGINX_PORT,http://127.0.0.1:$NGINX_PORT,http://$SERVER_IP:$NGINX_PORT,http://$SERVER_IP

# Performance
RATE_LIMIT_CALLS=1000
RATE_LIMIT_PERIOD=60

# Redis (Optional)
$([ "$REDIS_ENABLED" = true ] && echo "REDIS_URL=redis://localhost:$REDIS_PORT" || echo "#REDIS_URL=redis://localhost:$REDIS_PORT")
EOF

echo -e "${GREEN}✔ Backend .env ایجاد شد${NC}"

cd "$INSTALL_DIR"

################################################################################
# 9. نصب و Build Frontend
################################################################################

log_step "🎨 نصب و Build Frontend..."

cd "$INSTALL_DIR/frontend"

# پاکسازی
rm -rf node_modules build .cache 2>/dev/null || true

# نصب dependencies
echo -e "${YELLOW}⏳ نصب Node packages...${NC}"

if ! yarn install; then
    echo -e "${RED}❌ خطا در yarn install${NC}"
    exit 1
fi

echo -e "${GREEN}✔ Frontend dependencies نصب شد${NC}"

# ایجاد .env
cat > .env << EOF
REACT_APP_BACKEND_URL=http://$SERVER_IP:$NGINX_PORT
REACT_APP_API_URL=http://$SERVER_IP:$NGINX_PORT/api
EOF

echo -e "${GREEN}✔ Frontend .env ایجاد شد${NC}"

# Build برای production
echo -e "${YELLOW}⏳ Build کردن Frontend برای Production...${NC}"

if yarn build > /dev/null 2>&1; then
    echo -e "${GREEN}✔ Frontend build شد${NC}"
    BUILD_CREATED=true
else
    echo -e "${RED}❌ Build failed${NC}"
    exit 1
fi

cd "$INSTALL_DIR"

################################################################################
# 10. ایجاد PM2 Ecosystem File
################################################################################

log_step "⚡ ایجاد PM2 Configuration..."

cat > ecosystem.config.js << 'EOF'
module.exports = {
  apps: [
    {
      name: 'mlm-backend',
      script: 'venv/bin/python',
      args: 'server.py',
      cwd: './backend',
      interpreter: 'none',
      instances: 1,
      autorestart: true,
      watch: false,
      max_memory_restart: '1G',
      env: {
        NODE_ENV: 'production'
      },
      error_file: './backend/logs/pm2-error.log',
      out_file: './backend/logs/pm2-out.log',
      log_file: './backend/logs/pm2-combined.log',
      time: true
    },
    {
      name: 'mlm-frontend',
      script: 'npx',
      args: 'serve -s build -l 3000',
      cwd: './frontend',
      instances: 1,
      autorestart: true,
      watch: false,
      max_memory_restart: '500M',
      env: {
        NODE_ENV: 'production'
      },
      error_file: './backend/logs/frontend-error.log',
      out_file: './backend/logs/frontend-out.log',
      log_file: './backend/logs/frontend-combined.log',
      time: true
    }
  ]
};
EOF

echo -e "${GREEN}✔ PM2 ecosystem.config.js ایجاد شد${NC}"

# نصب serve برای frontend
echo -e "${YELLOW}⏳ نصب serve برای سرو کردن Frontend...${NC}"
$SUDO_CMD npm install -g serve
echo -e "${GREEN}✔ serve نصب شد${NC}"

################################################################################
# 11. پیکربندی Nginx
################################################################################

log_step "🌐 پیکربندی Nginx..."

# ایجاد nginx config
cat > /tmp/mlm-nginx.conf << EOF
server {
    listen $NGINX_PORT;
    server_name _;
    
    client_max_body_size 50M;
    
    # Frontend - Static Files
    location / {
        proxy_pass http://127.0.0.1:$FRONTEND_PORT;
        proxy_http_version 1.1;
        proxy_set_header Upgrade \$http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host \$host;
        proxy_cache_bypass \$http_upgrade;
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto \$scheme;
    }
    
    # Backend API
    location /api {
        proxy_pass http://127.0.0.1:$BACKEND_PORT;
        proxy_http_version 1.1;
        proxy_set_header Upgrade \$http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host \$host;
        proxy_cache_bypass \$http_upgrade;
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto \$scheme;
        
        # Timeouts برای API های طولانی
        proxy_connect_timeout 60s;
        proxy_send_timeout 60s;
        proxy_read_timeout 60s;
    }
    
    # Health check
    location /health {
        proxy_pass http://127.0.0.1:$BACKEND_PORT/api/health;
        access_log off;
    }
}
EOF

# نصب nginx config
$SUDO_CMD cp /tmp/mlm-nginx.conf /etc/nginx/sites-available/mlm
$SUDO_CMD rm -f /etc/nginx/sites-enabled/default
$SUDO_CMD ln -sf /etc/nginx/sites-available/mlm /etc/nginx/sites-enabled/mlm

# تست nginx config
if $SUDO_CMD nginx -t > /dev/null 2>&1; then
    echo -e "${GREEN}✔ Nginx configuration معتبر است${NC}"
    $SUDO_CMD systemctl restart nginx
    $SUDO_CMD systemctl enable nginx
    echo -e "${GREEN}✔ Nginx راه‌اندازی شد${NC}"
else
    echo -e "${RED}❌ Nginx configuration نامعتبر است${NC}"
    $SUDO_CMD nginx -t
    exit 1
fi

################################################################################
# 12. ایجاد پوشه logs
################################################################################

echo -e "${YELLOW}⏳ ایجاد پوشه logs...${NC}"
mkdir -p backend/logs
echo -e "${GREEN}✔ پوشه logs ایجاد شد${NC}"

################################################################################
# 13. راه‌اندازی سرویس‌ها با PM2
################################################################################

log_step "🚀 راه‌اندازی سرویس‌ها..."

echo -e "${YELLOW}⏳ شروع Backend و Frontend با PM2...${NC}"

# Start services with PM2
pm2 start ecosystem.config.js
pm2 save

# انتظار برای راه‌اندازی
sleep 5

# چک کردن وضعیت
if pm2 list | grep -q "mlm-backend.*online"; then
    echo -e "${GREEN}✔ Backend در حال اجرا است${NC}"
    BACKEND_STATUS="✅ Running"
else
    echo -e "${RED}✘ Backend راه‌اندازی نشد${NC}"
    BACKEND_STATUS="❌ Failed"
    pm2 logs mlm-backend --lines 20 --nostream
fi

if pm2 list | grep -q "mlm-frontend.*online"; then
    echo -e "${GREEN}✔ Frontend در حال اجرا است${NC}"
    FRONTEND_STATUS="✅ Running"
else
    echo -e "${RED}✘ Frontend راه‌اندازی نشد${NC}"
    FRONTEND_STATUS="❌ Failed"
    pm2 logs mlm-frontend --lines 20 --nostream
fi

################################################################################
# 14. اعمال Database Indexes
################################################################################

log_step "📊 اعمال Database Indexes..."

echo -e "${YELLOW}⏳ اعمال indexes برای بهینه‌سازی...${NC}"

# ایجاد اسکریپت Python برای اعمال indexes
cat > /tmp/apply_indexes.py << 'PYEOF'
import asyncio
from motor.motor_asyncio import AsyncIOMotorClient
import os

async def apply_indexes():
    client = AsyncIOMotorClient("mongodb://localhost:27017")
    db = client[os.environ.get("DB_NAME", "mlm_production")]
    
    print("📊 اعمال indexes...")
    
    # Users indexes
    await db.users.create_index("user_id", unique=True)
    await db.users.create_index("email", unique=True)
    await db.users.create_index("referral_code", unique=True)
    await db.users.create_index("status")
    await db.users.create_index("depth")
    await db.users.create_index("referrer_id")
    await db.users.create_index("placement_parent_id")
    await db.users.create_index([("status", 1), ("depth", 1)])
    
    # Materialized Path indexes (Critical)
    await db.users.create_index("placement_path")
    await db.users.create_index([("placement_path", 1), ("status", 1)])
    await db.users.create_index([("placement_path", 1), ("depth", 1)])
    await db.users.create_index([("status", 1), ("direct_children_count", 1), ("depth", 1)])
    
    # Transactions indexes
    await db.transactions.create_index("user_id")
    await db.transactions.create_index("from_user_id")
    
    # Withdrawals indexes
    await db.withdrawals.create_index("user_id")
    await db.withdrawals.create_index("status")
    
    # Reward locks index
    await db.reward_locks.create_index("lock_id", unique=True)
    
    print("✅ Indexes اعمال شد")
    client.close()

if __name__ == "__main__":
    asyncio.run(apply_indexes())
PYEOF

# اجرای اسکریپت
cd "$INSTALL_DIR/backend"
source venv/bin/activate
export DB_NAME=$DB_NAME
python /tmp/apply_indexes.py
deactivate
cd "$INSTALL_DIR"

echo -e "${GREEN}✔ Database indexes اعمال شد${NC}"

################################################################################
# نمایش اطلاعات نهایی
################################################################################

clear
cat << "EOF"

════════════════════════════════════════════════════════════════════════
          ✅ نصب با موفقیت کامل شد! 🎉
════════════════════════════════════════════════════════════════════════

EOF

echo -e "${CYAN}╔══════════════════════════════════════════════════════════════════════╗${NC}"
echo -e "${CYAN}║${NC}                     ${BOLD}وضعیت سیستم${NC}                                   ${CYAN}║${NC}"
echo -e "${CYAN}╚══════════════════════════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "   MongoDB:     ✅ Running (Docker)"
echo -e "   Redis:       $([ \"$REDIS_ENABLED\" = true ] && echo '✅ Running (Docker)' || echo '○ Disabled')"
echo -e "   Backend:     $BACKEND_STATUS (PM2)"
echo -e "   Frontend:    $FRONTEND_STATUS (PM2)"
echo -e "   Nginx:       ✅ Running (Reverse Proxy)"
echo ""

echo -e "${CYAN}╔══════════════════════════════════════════════════════════════════════╗${NC}"
echo -e "${CYAN}║${NC}                   ${BOLD}🔐 اطلاعات ورود${NC}                                  ${CYAN}║${NC}"
echo -e "${CYAN}╚══════════════════════════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "${BOLD}${GREEN}   Email:    admin@mlm.com${NC}"
echo -e "${BOLD}${GREEN}   Password: admin123${NC}"
echo ""
echo -e "${RED}   ⚠️  حتماً رمز را بعد از اولین ورود تغییر دهید!${NC}"
echo ""

echo -e "${CYAN}╔══════════════════════════════════════════════════════════════════════╗${NC}"
echo -e "${CYAN}║${NC}                   ${BOLD}🌐 دسترسی به سیستم${NC}                              ${CYAN}║${NC}"
echo -e "${CYAN}╚══════════════════════════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "${GREEN}   🌍 آدرس اصلی:      http://$SERVER_IP${NC}"
echo -e "${GREEN}   📱 Local:          http://localhost${NC}"
echo -e "${GREEN}   🔗 API:            http://$SERVER_IP/api${NC}"
echo -e "${GREEN}   💚 Health Check:   http://$SERVER_IP/health${NC}"
echo ""

echo -e "${CYAN}╔══════════════════════════════════════════════════════════════════════╗${NC}"
echo -e "${CYAN}║${NC}                   ${BOLD}🛠️  مدیریت سرویس‌ها${NC}                             ${CYAN}║${NC}"
echo -e "${CYAN}╚══════════════════════════════════════════════════════════════════════╝${NC}"
echo ""

echo -e "${YELLOW}PM2 Commands:${NC}"
echo -e "${CYAN}   pm2 list                      ${NC}# مشاهده وضعیت"
echo -e "${CYAN}   pm2 logs mlm-backend          ${NC}# لاگ Backend"
echo -e "${CYAN}   pm2 logs mlm-frontend         ${NC}# لاگ Frontend"
echo -e "${CYAN}   pm2 restart mlm-backend       ${NC}# ریستارت Backend"
echo -e "${CYAN}   pm2 restart mlm-frontend      ${NC}# ریستارت Frontend"
echo -e "${CYAN}   pm2 stop all                  ${NC}# توقف همه"
echo -e "${CYAN}   pm2 start all                 ${NC}# شروع همه"
echo ""

echo -e "${YELLOW}Docker Commands:${NC}"
echo -e "${CYAN}   sudo docker ps                ${NC}# مشاهده containers"
echo -e "${CYAN}   sudo docker logs mlm-mongodb  ${NC}# لاگ MongoDB"
echo -e "${CYAN}   sudo docker restart mlm-mongodb ${NC}# ریستارت MongoDB"
echo ""

echo -e "${YELLOW}Nginx Commands:${NC}"
echo -e "${CYAN}   sudo systemctl status nginx   ${NC}# وضعیت"
echo -e "${CYAN}   sudo systemctl restart nginx  ${NC}# ریستارت"
echo -e "${CYAN}   sudo nginx -t                 ${NC}# تست config"
echo ""

echo -e "${YELLOW}Firewall (اگر فعال است):${NC}"
echo -e "${CYAN}   sudo ufw allow 80${NC}"
echo -e "${CYAN}   sudo ufw allow 443${NC}"
echo ""

echo "════════════════════════════════════════════════════════════════════════"
echo -e "${BOLD}${GREEN}         ✨ سیستم آماده استفاده است! Auto-start فعال است ✨${NC}"
echo "════════════════════════════════════════════════════════════════════════"
echo ""
echo -e "${BLUE}📍 بعد از Reboot، تمام سرویس‌ها خودکار شروع می‌شوند${NC}"
echo -e "${BLUE}📍 Log نصب: $LOG_FILE${NC}"
echo ""
