import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { toast } from 'sonner';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { DollarSign, Clock, CheckCircle, XCircle, Wallet, AlertCircle } from 'lucide-react';

const API = process.env.REACT_APP_BACKEND_URL || '';

export default function WithdrawalPage({ user }) {
  const [withdrawals, setWithdrawals] = useState([]);
  const [amount, setAmount] = useState('');
  const [cryptoAddress, setCryptoAddress] = useState('');
  const [network, setNetwork] = useState('USDT-TRC20');
  const [loading, setLoading] = useState(false);
  const [enabledNetworks, setEnabledNetworks] = useState([]);
  const [networksLoading, setNetworksLoading] = useState(true);
  const [minWithdrawal, setMinWithdrawal] = useState(10);

  useEffect(() => {
    fetchWithdrawals();
    fetchEnabledNetworks();
    fetchWithdrawalSettings();
  }, []);

  const fetchWithdrawals = async () => {
    try {
      const token = localStorage.getItem('token');
      const res = await axios.get(`${API}/api/user/withdrawals`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setWithdrawals(res.data);
    } catch (error) {
      toast.error('خطا در دریافت تاریخچه');
      console.error(error);
    }
  };

  const fetchEnabledNetworks = async () => {
    setNetworksLoading(true);
    try {
      const res = await axios.get(`${API}/api/user/crypto-payment/networks`);
      console.log('Fetched networks:', res.data);
      
      if (res.data && Array.isArray(res.data) && res.data.length > 0) {
        setEnabledNetworks(res.data);
        setNetwork(res.data[0].value);
      } else {
        // اگر هیچ شبکه‌ای فعال نباشد، شبکه پیش‌فرض را تنظیم کن
        console.warn('هیچ شبکه فعالی دریافت نشد، استفاده از پیش‌فرض');
        setEnabledNetworks([{
          value: 'USDT-TRC20',
          label: 'USDT-TRC20 (TRON)',
          name: 'TRON'
        }]);
        setNetwork('USDT-TRC20');
      }
    } catch (error) {
      console.error('خطا در دریافت شبکه‌های فعال:', error);
      // در صورت خطا، شبکه پیش‌فرض را تنظیم کن
      setEnabledNetworks([{
        value: 'USDT-TRC20',
        label: 'USDT-TRC20 (TRON)',
        name: 'TRON'
      }]);
      setNetwork('USDT-TRC20');
    } finally {
      setNetworksLoading(false);
    }
  };

  const fetchWithdrawalSettings = async () => {
    try {
      const res = await axios.get(`${API}/api/user/withdrawal-settings`);
      if (res.data && res.data.min_withdrawal_amount) {
        setMinWithdrawal(res.data.min_withdrawal_amount);
      }
    } catch (error) {
      console.error('خطا در دریافت تنظیمات برداشت:', error);
      // اگر خطا داشت، از مقدار پیش‌فرض 10 استفاده می‌کند
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();

    if (parseFloat(amount) < minWithdrawal) {
      toast.error(`حداقل مبلغ برداشت ${minWithdrawal} USDT است`);
      return;
    }

    if (parseFloat(amount) > user.total_points) {
      toast.error('موجودی کافی نیست');
      return;
    }

    if (!cryptoAddress.trim()) {
      toast.error('آدرس کیف پول را وارد کنید');
      return;
    }

    // اعتبارسنجی آدرس بر اساس شبکه
    if ((network === 'USDT-TRC20' || network === 'TRON') && !cryptoAddress.startsWith('T')) {
      toast.error('آدرس TRON باید با T شروع شود');
      return;
    }
    
    if ((network === 'USDT-BEP20' || network === 'BSC') && !cryptoAddress.startsWith('0x')) {
      toast.error('آدرس BSC باید با 0x شروع شود');
      return;
    }

    setLoading(true);
    try {
      const token = localStorage.getItem('token');
      await axios.post(`${API}/api/user/withdrawals`, {
        amount: parseFloat(amount),
        crypto_address: cryptoAddress,
        network: network
      }, {
        headers: { Authorization: `Bearer ${token}` }
      });
      toast.success('✅ درخواست برداشت ثبت شد');
      setAmount('');
      setCryptoAddress('');
      fetchWithdrawals();
    } catch (error) {
      toast.error(error.response?.data?.detail || 'خطا در ثبت درخواست');
    } finally {
      setLoading(false);
    }
  };

  const getStatusBadge = (status) => {
    const map = {
      pending_admin: { label: 'در انتظار تایید', icon: Clock, color: 'bg-amber-100 text-amber-700' },
      completed: { label: 'تکمیل شده', icon: CheckCircle, color: 'bg-green-100 text-green-700' },
      rejected: { label: 'رد شده', icon: XCircle, color: 'bg-red-100 text-red-700' }
    };
    const s = map[status] || map.pending_admin;
    const Icon = s.icon;
    return (
      <span className={`px-3 py-1 rounded-full text-sm flex items-center gap-1 ${s.color}`}>
        <Icon className="w-4 h-4" />
        {s.label}
      </span>
    );
  };

  if (networksLoading) {
    return (
      <div className="p-6 flex items-center justify-center">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500"></div>
        <span className="mr-3">در حال بارگذاری...</span>
      </div>
    );
  }

  return (
    <div className="p-6 space-y-6 max-w-4xl mx-auto">
      {/* فرم درخواست برداشت */}
      <Card className="shadow-lg border-blue-200">
        <CardHeader className="bg-gradient-to-r from-blue-500 to-indigo-600 text-white">
          <CardTitle className="text-2xl flex items-center gap-2">
            <Wallet className="w-7 h-7" />
            درخواست برداشت وجه
          </CardTitle>
        </CardHeader>
        <CardContent className="p-6">
          <div className="mb-6 p-5 bg-gradient-to-br from-blue-50 to-indigo-50 rounded-xl border-2 border-blue-200">
            <div className="flex items-center justify-between">
              <div>
                <div className="text-sm text-gray-600 mb-1">💰 موجودی قابل برداشت:</div>
                <div className="text-4xl font-bold text-blue-700">
                  {user?.total_points?.toFixed(2) || '0.00'} <span className="text-xl">USDT</span>
                </div>
              </div>
              <DollarSign className="w-16 h-16 text-blue-300 opacity-50" />
            </div>
          </div>

          <form onSubmit={handleSubmit} className="space-y-5">
            <div>
              <Label className="text-base font-semibold">مبلغ برداشت (حداقل {minWithdrawal} USDT)</Label>
              <Input
                type="number"
                step="0.01"
                value={amount}
                onChange={(e) => setAmount(e.target.value)}
                placeholder="مثال: 50"
                className="mt-2 text-lg p-3"
                required
              />
            </div>

            <div>
              <Label className="text-base font-semibold">آدرس کیف پول</Label>
              <Input
                type="text"
                value={cryptoAddress}
                onChange={(e) => setCryptoAddress(e.target.value)}
                placeholder={network === 'USDT-TRC20' || network === 'TRON' ? 'مثال: TXyz123...' : 'مثال: 0x123...'}
                className="mt-2 font-mono text-sm p-3"
                required
              />
              <p className="text-xs text-gray-500 mt-1">
                {network === 'USDT-TRC20' || network === 'TRON' 
                  ? '💡 آدرس TRON باید با حرف T شروع شود و 34 کاراکتر باشد'
                  : '💡 آدرس BSC باید با 0x شروع شود و 42 کاراکتر باشد'}
              </p>
            </div>

            <div>
              <Label className="text-base font-semibold">شبکه</Label>
              <select
                value={network}
                onChange={(e) => setNetwork(e.target.value)}
                className="w-full border-2 border-gray-300 rounded-lg p-3 mt-2 focus:outline-none focus:border-blue-500"
              >
                {enabledNetworks.map((net) => (
                  <option key={net.value} value={net.value}>
                    {net.label}
                  </option>
                ))}
              </select>
            </div>

            {/* هشدار */}
            <div className="bg-amber-50 border-2 border-amber-300 p-4 rounded-lg flex gap-3">
              <AlertCircle className="w-5 h-5 text-amber-600 flex-shrink-0 mt-0.5" />
              <div className="text-sm text-amber-800">
                <strong>توجه:</strong> لطفاً آدرس کیف پول خود را با دقت وارد کنید. 
                ارسال به آدرس اشتباه قابل بازگشت نیست.
              </div>
            </div>

            <Button 
              type="submit" 
              disabled={loading} 
              className="w-full text-lg py-6 bg-blue-600 hover:bg-blue-700"
            >
              {loading ? (
                <>
                  <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white mr-2"></div>
                  در حال ارسال...
                </>
              ) : (
                '✅ ثبت درخواست برداشت'
              )}
            </Button>
          </form>
        </CardContent>
      </Card>

      {/* تاریخچه برداشت‌ها */}
      <Card>
        <CardHeader>
          <CardTitle>تاریخچه برداشت‌ها</CardTitle>
        </CardHeader>
        <CardContent>
          {withdrawals.length === 0 ? (
            <div className="text-center py-8 text-gray-500">هیچ درخواستی ثبت نشده است</div>
          ) : (
            <div className="space-y-3">
              {withdrawals.map((w) => (
                <div key={w.withdrawal_id} className="border rounded-lg p-4 hover:shadow-md transition">
                  <div className="flex justify-between items-start mb-2">
                    <div>
                      <div className="font-semibold text-lg">{w.amount} USDT</div>
                      <div className="text-xs text-gray-500">{w.network}</div>
                    </div>
                    {getStatusBadge(w.status)}
                  </div>
                  <div className="text-sm text-gray-600 mt-2">
                    <div className="font-mono text-xs break-all">{w.crypto_address}</div>
                    <div className="mt-1">تاریخ: {new Date(w.requested_at).toLocaleDateString('fa-IR')}</div>
                    {w.transaction_hash && (
                      <div className="mt-1 text-green-600">
                        Hash: {w.transaction_hash.substring(0, 20)}...
                      </div>
                    )}
                    {w.admin_note && (
                      <div className="mt-1 text-red-600">توضیح: {w.admin_note}</div>
                    )}
                  </div>
                </div>
              ))}
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
