"""Reports Routes - سیستم گزارشات جامع ادمین

این فایل شامل تمام گزارشات مالی و تحلیلی سیستم است.
- گزارشات فقط On-Demand اجرا میشن (بدون cache)
- هیچ تأثیری روی عملکرد سیستم اصلی ندارن
- فقط برای ادمین قابل دسترسی هستن
"""

import logging
from datetime import datetime, timezone, timedelta
from typing import Optional

from fastapi import APIRouter, HTTPException, Depends, Query
from services.auth_service import get_admin_user
from database import get_db

router = APIRouter()


# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
# گزارشات پاداش‌ها
# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

@router.get("/admin/reports/rewards")
async def get_rewards_report(
    reward_type: Optional[str] = None,
    page: int = Query(1, ge=1),
    limit: int = Query(50, ge=1, le=200),
    admin_user: dict = Depends(get_admin_user)
):
    """
    گزارش کامل پاداش‌ها
    
    - reward_type: direct_referral, level_1, level_2, ... (optional)
    - نمایش: چه کسی، چقدر، از کی، چرا
    """
    db = await get_db()
    
    # ساخت فیلتر
    filter_query = {}
    if reward_type:
        filter_query["reward_type"] = reward_type
    
    # شمارش کل
    total = await db.transactions.count_documents(filter_query)
    
    # دریافت تراکنش‌ها با pagination
    skip = (page - 1) * limit
    transactions = await db.transactions.find(
        filter_query,
        {"_id": 0}
    ).sort("timestamp", -1).skip(skip).limit(limit).to_list(limit)
    
    # محاسبه آمار کلی (بدون pagination)
    stats_pipeline = [
        {"$match": filter_query},
        {
            "$group": {
                "_id": "$reward_type",
                "total_amount": {"$sum": "$amount"},
                "count": {"$sum": 1}
            }
        },
        {"$sort": {"total_amount": -1}}
    ]
    
    stats = await db.transactions.aggregate(stats_pipeline).to_list(100)
    
    # جمع کل پاداش‌ها
    total_rewards = sum(s["total_amount"] for s in stats)
    
    return {
        "transactions": transactions,
        "total": total,
        "page": page,
        "pages": (total + limit - 1) // limit,
        "summary": {
            "total_rewards": total_rewards,
            "by_type": stats
        }
    }


@router.get("/admin/reports/rewards/top-earners")
async def get_top_earners(
    limit: int = Query(20, ge=1, le=100),
    admin_user: dict = Depends(get_admin_user)
):
    """
    کاربران پردرآمد
    
    - بیشترین پاداش گیرندگان
    - مرتب شده بر اساس total_points
    """
    db = await get_db()
    
    top_earners = await db.users.find(
        {"status": "active", "total_points": {"$gt": 0}},
        {"_id": 0, "user_id": 1, "email": 1, "total_points": 1, "depth": 1}
    ).sort("total_points", -1).limit(limit).to_list(limit)
    
    return {
        "top_earners": top_earners,
        "count": len(top_earners)
    }


@router.get("/admin/reports/rewards/by-user/{user_id}")
async def get_user_rewards_detail(
    user_id: str,
    page: int = Query(1, ge=1),
    limit: int = Query(50, ge=1, le=200),
    admin_user: dict = Depends(get_admin_user)
):
    """
    جزئیات کامل پاداش‌های یک کاربر
    
    - همه پاداش‌هایی که گرفته
    - از چه کسانی گرفته
    - چه نوع پاداشی
    """
    db = await get_db()
    
    # اطلاعات کاربر
    user = await db.users.find_one(
        {"user_id": user_id},
        {"_id": 0, "user_id": 1, "email": 1, "total_points": 1}
    )
    
    if not user:
        raise HTTPException(status_code=404, detail="کاربر یافت نشد")
    
    # تراکنش‌های کاربر
    filter_query = {"user_id": user_id}
    total = await db.transactions.count_documents(filter_query)
    
    skip = (page - 1) * limit
    transactions = await db.transactions.find(
        filter_query,
        {"_id": 0}
    ).sort("timestamp", -1).skip(skip).limit(limit).to_list(limit)
    
    # آمار تفکیک شده
    stats = await db.transactions.aggregate([
        {"$match": {"user_id": user_id}},
        {
            "$group": {
                "_id": "$reward_type",
                "total": {"$sum": "$amount"},
                "count": {"$sum": 1}
            }
        }
    ]).to_list(100)
    
    return {
        "user": user,
        "transactions": transactions,
        "total": total,
        "page": page,
        "pages": (total + limit - 1) // limit,
        "stats_by_type": stats
    }


# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
# گزارشات مالی
# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

@router.get("/admin/reports/financial/summary")
async def get_financial_summary(admin_user: dict = Depends(get_admin_user)):
    """
    خلاصه مالی سیستم
    
    - کل واریزها (payments)
    - کل برداشت‌ها (withdrawals)
    - بالانس کل سیستم
    - سود خالص
    """
    db = await get_db()
    
    # کل پرداخت‌های تأیید شده
    payments_pipeline = [
        {"$match": {"status": "confirmed"}},
        {"$group": {"_id": None, "total": {"$sum": "$amount"}}}
    ]
    payments_result = await db.crypto_payments.aggregate(payments_pipeline).to_list(1)
    total_payments = payments_result[0]["total"] if payments_result else 0
    
    # کل برداشت‌های تأیید شده
    withdrawals_pipeline = [
        {"$match": {"status": "completed"}},
        {"$group": {"_id": None, "total": {"$sum": "$amount"}}}
    ]
    withdrawals_result = await db.withdrawals.aggregate(withdrawals_pipeline).to_list(1)
    total_withdrawals = withdrawals_result[0]["total"] if withdrawals_result else 0
    
    # کل پاداش‌های پرداخت شده
    rewards_pipeline = [
        {"$group": {"_id": None, "total": {"$sum": "$amount"}}}
    ]
    rewards_result = await db.transactions.aggregate(rewards_pipeline).to_list(1)
    total_rewards = rewards_result[0]["total"] if rewards_result else 0
    
    # مجموع موجودی کاربران
    users_balance_pipeline = [
        {"$group": {"_id": None, "total": {"$sum": "$total_points"}}}
    ]
    users_balance_result = await db.users.aggregate(users_balance_pipeline).to_list(1)
    total_user_balance = users_balance_result[0]["total"] if users_balance_result else 0
    
    # محاسبات
    net_profit = total_payments - total_withdrawals
    system_balance = total_payments - total_rewards
    
    return {
        "total_payments": total_payments,
        "total_withdrawals": total_withdrawals,
        "total_rewards": total_rewards,
        "total_user_balance": total_user_balance,
        "net_profit": net_profit,
        "system_balance": system_balance,
        "generated_at": datetime.now(timezone.utc).isoformat()
    }


@router.get("/admin/reports/financial/payments")
async def get_payments_report(
    status: Optional[str] = None,
    network: Optional[str] = None,
    page: int = Query(1, ge=1),
    limit: int = Query(50, ge=1, le=200),
    admin_user: dict = Depends(get_admin_user)
):
    """
    گزارش پرداخت‌های کریپتو
    
    - وضعیت: pending, confirmed, expired
    - شبکه: TRON, BSC
    - با pagination
    """
    db = await get_db()
    
    # ساخت فیلتر
    filter_query = {}
    if status:
        filter_query["status"] = status
    if network:
        filter_query["network"] = network.upper()
    
    # شمارش کل
    total = await db.crypto_payments.count_documents(filter_query)
    
    # دریافت پرداخت‌ها
    skip = (page - 1) * limit
    payments = await db.crypto_payments.find(
        filter_query,
        {"_id": 0}
    ).sort("created_at", -1).skip(skip).limit(limit).to_list(limit)
    
    # آمار
    stats = await db.crypto_payments.aggregate([
        {"$match": filter_query},
        {
            "$group": {
                "_id": "$status",
                "count": {"$sum": 1},
                "total_amount": {"$sum": "$amount"}
            }
        }
    ]).to_list(100)
    
    return {
        "payments": payments,
        "total": total,
        "page": page,
        "pages": (total + limit - 1) // limit,
        "stats": stats
    }


@router.get("/admin/reports/financial/withdrawals")
async def get_withdrawals_report(
    status: Optional[str] = None,
    page: int = Query(1, ge=1),
    limit: int = Query(50, ge=1, le=200),
    admin_user: dict = Depends(get_admin_user)
):
    """
    گزارش برداشت‌ها
    
    - وضعیت: pending_admin, completed, rejected
    - شامل کارمزد
    """
    db = await get_db()
    
    # ساخت فیلتر
    filter_query = {}
    if status:
        filter_query["status"] = status
    
    # شمارش کل
    total = await db.withdrawals.count_documents(filter_query)
    
    # دریافت برداشت‌ها
    skip = (page - 1) * limit
    withdrawals = await db.withdrawals.find(
        filter_query,
        {"_id": 0}
    ).sort("requested_at", -1).skip(skip).limit(limit).to_list(limit)
    
    # آمار
    stats = await db.withdrawals.aggregate([
        {"$match": filter_query},
        {
            "$group": {
                "_id": "$status",
                "count": {"$sum": 1},
                "total_amount": {"$sum": "$amount"},
                "total_fee": {"$sum": "$fee"}
            }
        }
    ]).to_list(100)
    
    return {
        "withdrawals": withdrawals,
        "total": total,
        "page": page,
        "pages": (total + limit - 1) // limit,
        "stats": stats
    }


# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
# گزارشات کاربران
# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

@router.get("/admin/reports/users/top-referrers")
async def get_top_referrers(
    limit: int = Query(20, ge=1, le=100),
    admin_user: dict = Depends(get_admin_user)
):
    """
    کاربران با بیشترین دعوت مستقیم
    """
    db = await get_db()
    
    # محاسبه تعداد referrals برای هر کاربر
    pipeline = [
        {"$match": {"status": "active", "referrer_id": {"$ne": None}}},
        {
            "$group": {
                "_id": "$referrer_id",
                "direct_referrals_count": {"$sum": 1}
            }
        },
        {"$sort": {"direct_referrals_count": -1}},
        {"$limit": limit}
    ]
    
    top_referrers_data = await db.users.aggregate(pipeline).to_list(limit)
    
    # دریافت اطلاعات کاربران
    referrer_ids = [item["_id"] for item in top_referrers_data]
    users = await db.users.find(
        {"user_id": {"$in": referrer_ids}},
        {"_id": 0, "user_id": 1, "email": 1, "total_points": 1}
    ).to_list(limit)
    
    # ترکیب داده‌ها
    users_dict = {u["user_id"]: u for u in users}
    
    top_referrers = []
    for item in top_referrers_data:
        user_info = users_dict.get(item["_id"], {})
        top_referrers.append({
            "user_id": item["_id"],
            "email": user_info.get("email"),
            "direct_referrals": item["direct_referrals_count"],
            "total_points": user_info.get("total_points", 0)
        })
    
    return {
        "top_referrers": top_referrers,
        "count": len(top_referrers)
    }


@router.get("/admin/reports/users/tree-stats")
async def get_tree_stats(admin_user: dict = Depends(get_admin_user)):
    """
    آمار کلی درخت
    
    - تعداد کاربران در هر depth
    - میانگین زیرمجموعه‌ها
    - کاربران با بیشترین زیردرخت
    """
    db = await get_db()
    
    # آمار بر اساس depth
    depth_pipeline = [
        {"$match": {"status": "active"}},
        {
            "$group": {
                "_id": "$depth",
                "count": {"$sum": 1},
                "avg_children": {"$avg": "$direct_children_count"}
            }
        },
        {"$sort": {"_id": 1}}
    ]
    
    depth_stats = await db.users.aggregate(depth_pipeline).to_list(10)
    
    # کاربران با بیشترین زیردرخت
    top_trees = await db.users.find(
        {"status": "active", "total_descendants": {"$gt": 0}},
        {"_id": 0, "user_id": 1, "email": 1, "total_descendants": 1, "depth": 1}
    ).sort("total_descendants", -1).limit(10).to_list(10)
    
    # کل کاربران
    total_users = await db.users.count_documents({"status": "active"})
    
    return {
        "total_active_users": total_users,
        "by_depth": depth_stats,
        "top_trees": top_trees
    }


@router.get("/admin/reports/users/activation-timeline")
async def get_activation_timeline(
    days: int = Query(30, ge=1, le=365),
    admin_user: dict = Depends(get_admin_user)
):
    """
    نمودار فعال‌سازی کاربران در طول زمان
    
    - تعداد کاربران فعال شده در هر روز
    - برای days روز اخیر
    """
    db = await get_db()
    
    # محاسبه تاریخ شروع
    end_date = datetime.now(timezone.utc)
    start_date = end_date - timedelta(days=days)
    
    # گروه‌بندی بر اساس روز
    pipeline = [
        {
            "$match": {
                "status": "active",
                "activated_at": {"$gte": start_date.isoformat()}
            }
        },
        {
            "$group": {
                "_id": {
                    "$substr": ["$activated_at", 0, 10]  # YYYY-MM-DD
                },
                "count": {"$sum": 1}
            }
        },
        {"$sort": {"_id": 1}}
    ]
    
    timeline = await db.users.aggregate(pipeline).to_list(days)
    
    return {
        "timeline": [
            {"date": item["_id"], "activations": item["count"]}
            for item in timeline
        ],
        "total_activations": sum(item["count"] for item in timeline),
        "period_days": days
    }


# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
# گزارش جامع سیستم
# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

@router.get("/admin/reports/system/health")
async def get_system_health(admin_user: dict = Depends(get_admin_user)):
    """
    وضعیت کلی سیستم
    
    - همه شاخص‌های کلیدی در یک نگاه
    - برای monitoring و بررسی سریع
    """
    db = await get_db()
    
    # کاربران
    total_users = await db.users.count_documents({})
    active_users = await db.users.count_documents({"status": "active"})
    pending_users = await db.users.count_documents({"status": "pending"})
    
    # پرداخت‌ها
    pending_payments = await db.crypto_payments.count_documents({"status": "pending"})
    confirmed_payments = await db.crypto_payments.count_documents({"status": "confirmed"})
    
    # برداشت‌ها
    pending_withdrawals = await db.withdrawals.count_documents({"status": "pending_admin"})
    completed_withdrawals = await db.withdrawals.count_documents({"status": "completed"})
    
    # تراکنش‌ها
    total_transactions = await db.transactions.count_documents({})
    
    # مالی
    total_rewards_result = await db.transactions.aggregate([
        {"$group": {"_id": None, "total": {"$sum": "$amount"}}}
    ]).to_list(1)
    total_rewards = total_rewards_result[0]["total"] if total_rewards_result else 0
    
    total_balance_result = await db.users.aggregate([
        {"$group": {"_id": None, "total": {"$sum": "$total_points"}}}
    ]).to_list(1)
    total_user_balance = total_balance_result[0]["total"] if total_balance_result else 0
    
    return {
        "users": {
            "total": total_users,
            "active": active_users,
            "pending": pending_users
        },
        "payments": {
            "pending": pending_payments,
            "confirmed": confirmed_payments
        },
        "withdrawals": {
            "pending": pending_withdrawals,
            "completed": completed_withdrawals
        },
        "transactions": {
            "total": total_transactions
        },
        "financial": {
            "total_rewards_paid": total_rewards,
            "total_user_balance": total_user_balance
        },
        "generated_at": datetime.now(timezone.utc).isoformat()
    }


# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
# گزارش تخصصی: تحلیل عملکرد
# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

@router.get("/admin/reports/performance/user-status/{user_id}")
async def get_user_performance(
    user_id: str,
    admin_user: dict = Depends(get_admin_user)
):
    """
    عملکرد کامل یک کاربر
    
    - همه چیز درباره این کاربر
    - برای debugging و بررسی دقیق
    """
    db = await get_db()
    
    # اطلاعات کاربر
    user = await db.users.find_one(
        {"user_id": user_id},
        {"_id": 0, "password": 0}
    )
    
    if not user:
        raise HTTPException(status_code=404, detail="کاربر یافت نشد")
    
    # تعداد referrals
    direct_referrals = await db.users.count_documents({"referrer_id": user_id})
    active_referrals = await db.users.count_documents({
        "referrer_id": user_id,
        "status": "active"
    })
    
    # پاداش‌های دریافتی
    rewards = await db.transactions.find(
        {"user_id": user_id},
        {"_id": 0}
    ).sort("timestamp", -1).limit(20).to_list(20)
    
    rewards_stats = await db.transactions.aggregate([
        {"$match": {"user_id": user_id}},
        {
            "$group": {
                "_id": "$reward_type",
                "total": {"$sum": "$amount"},
                "count": {"$sum": 1}
            }
        }
    ]).to_list(100)
    
    # پرداخت‌ها
    payments = await db.crypto_payments.find(
        {"user_id": user_id},
        {"_id": 0}
    ).sort("created_at", -1).limit(10).to_list(10)
    
    # برداشت‌ها
    withdrawals = await db.withdrawals.find(
        {"user_id": user_id},
        {"_id": 0}
    ).sort("requested_at", -1).limit(10).to_list(10)
    
    return {
        "user": user,
        "referrals": {
            "direct_count": direct_referrals,
            "active_count": active_referrals
        },
        "rewards": {
            "recent": rewards,
            "stats": rewards_stats
        },
        "payments": payments,
        "withdrawals": withdrawals
    }
