import { useState, useEffect } from "react";
import axios from "axios";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Users, TrendingUp, TreePine, RefreshCw, Loader2 } from "lucide-react";
import { toast } from "sonner";

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

export default function AdminSeedDashboard() {
  const [seedData, setSeedData] = useState(null);
  const [loading, setLoading] = useState(true);

  const fetchSeedDashboard = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem("token");
      const response = await axios.get(`${API}/admin/seed/dashboard`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setSeedData(response.data);
    } catch (error) {
      console.error("Error fetching seed dashboard:", error);
      toast.error("خطا در دریافت اطلاعات seed");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchSeedDashboard();
  }, []);

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <Loader2 className="w-8 h-8 animate-spin text-blue-600" />
      </div>
    );
  }

  if (!seedData) {
    return (
      <Card>
        <CardContent className="p-8 text-center">
          <p className="text-gray-500">اطلاعات seed یافت نشد</p>
        </CardContent>
      </Card>
    );
  }

  const { user_info, tree_stats, level_counts, recent_transactions, direct_children } = seedData;

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-bold text-gray-800">Dashboard Seed</h2>
          <p className="text-sm text-gray-500">نمای کاربری کاربر ریشه (seed)</p>
        </div>
        <Button onClick={fetchSeedDashboard} variant="outline" size="sm">
          <RefreshCw className="w-4 h-4 ml-2" />
          بروزرسانی
        </Button>
      </div>

      {/* User Info */}
      <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Users className="w-5 h-5 text-blue-600" />
            اطلاعات کاربری
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div>
              <p className="text-sm text-gray-600">شناسه کاربری</p>
              <p className="font-semibold text-gray-800">{user_info.user_id}</p>
            </div>
            <div>
              <p className="text-sm text-gray-600">ایمیل</p>
              <p className="font-semibold text-gray-800">{user_info.email}</p>
            </div>
            <div>
              <p className="text-sm text-gray-600">وضعیت</p>
              <Badge className="bg-green-600">{user_info.status}</Badge>
            </div>
            <div>
              <p className="text-sm text-gray-600">کد دعوت</p>
              <p className="font-semibold text-gray-800">{user_info.referral_code}</p>
            </div>
            <div>
              <p className="text-sm text-gray-600">امتیاز کل</p>
              <p className="font-semibold text-gray-800">{user_info.total_points.toFixed(2)}</p>
            </div>
            <div>
              <p className="text-sm text-gray-600">عمق در درخت</p>
              <p className="font-semibold text-gray-800">{user_info.depth}</p>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Tree Stats */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
          <CardHeader>
            <CardTitle className="text-lg flex items-center gap-2">
              <TreePine className="w-5 h-5 text-green-600" />
              فرزندان مستقیم
            </CardTitle>
          </CardHeader>
          <CardContent>
            <p className="text-3xl font-bold text-green-700">
              {tree_stats.direct_children_count}
            </p>
            <p className="text-sm text-gray-600 mt-1">تعداد در درخت placement</p>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
          <CardHeader>
            <CardTitle className="text-lg flex items-center gap-2">
              <Users className="w-5 h-5 text-purple-600" />
              کل زیردرخت
            </CardTitle>
          </CardHeader>
          <CardContent>
            <p className="text-3xl font-bold text-purple-700">
              {tree_stats.total_descendants}
            </p>
            <p className="text-sm text-gray-600 mt-1">همه نسل‌ها</p>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
          <CardHeader>
            <CardTitle className="text-lg flex items-center gap-2">
              <TrendingUp className="w-5 h-5 text-blue-600" />
              دعوت‌های فعال
            </CardTitle>
          </CardHeader>
          <CardContent>
            <p className="text-3xl font-bold text-blue-700">
              {tree_stats.direct_referrals}
            </p>
            <p className="text-sm text-gray-600 mt-1">referrals فعال</p>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
          <CardHeader>
            <CardTitle className="text-lg flex items-center gap-2">
              <Users className="w-5 h-5 text-orange-600" />
              در انتظار
            </CardTitle>
          </CardHeader>
          <CardContent>
            <p className="text-3xl font-bold text-orange-700">
              {tree_stats.pending_referrals}
            </p>
            <p className="text-sm text-gray-600 mt-1">referrals pending</p>
          </CardContent>
        </Card>
      </div>

      {/* Level Counts */}
      <Card>
        <CardHeader>
          <CardTitle>تعداد کاربران در هر سطح (7 سطح)</CardTitle>
          <CardDescription>نمودار توزیع کاربران در سطوح مختلف درخت</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-7 gap-4">
            {Object.entries(level_counts).map(([level, count]) => (
              <div
                key={level}
                className="bg-gradient-to-br from-indigo-50 to-indigo-100 p-4 rounded-lg text-center border border-indigo-200"
              >
                <p className="text-sm font-medium text-gray-600 mb-1">
                  {level.replace('level_', 'سطح ')}
                </p>
                <p className="text-2xl font-bold text-indigo-700">{count}</p>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* Recent Transactions */}
      <Card>
        <CardHeader>
          <CardTitle>تراکنش‌های اخیر ({recent_transactions.length})</CardTitle>
        </CardHeader>
        <CardContent>
          {recent_transactions.length === 0 ? (
            <p className="text-center text-gray-500 py-8">تراکنشی موجود نیست</p>
          ) : (
            <div className="space-y-2">
              {recent_transactions.slice(0, 10).map((tx, index) => (
                <div
                  key={index}
                  className="flex items-center justify-between p-3 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors"
                >
                  <div className="flex-1">
                    <p className="font-medium text-gray-800">{tx.reward_type}</p>
                    <p className="text-sm text-gray-500">
                      از: {tx.from_user_id || 'سیستم'}
                    </p>
                  </div>
                  <div className="text-left">
                    <p className="font-bold text-green-600">+{tx.amount.toFixed(2)}</p>
                    <p className="text-xs text-gray-500">
                      {new Date(tx.timestamp).toLocaleDateString('fa-IR')}
                    </p>
                  </div>
                </div>
              ))}
            </div>
          )}
        </CardContent>
      </Card>

      {/* Direct Children */}
      <Card>
        <CardHeader>
          <CardTitle>زیرمجموعه‌های مستقیم ({direct_children.length})</CardTitle>
        </CardHeader>
        <CardContent>
          {direct_children.length === 0 ? (
            <p className="text-center text-gray-500 py-8">زیرمجموعه‌ای موجود نیست</p>
          ) : (
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead>
                  <tr className="border-b">
                    <th className="text-right p-3 font-medium text-gray-600">ایمیل</th>
                    <th className="text-right p-3 font-medium text-gray-600">وضعیت</th>
                    <th className="text-right p-3 font-medium text-gray-600">امتیاز</th>
                    <th className="text-right p-3 font-medium text-gray-600">تاریخ ایجاد</th>
                  </tr>
                </thead>
                <tbody>
                  {direct_children.map((child) => (
                    <tr key={child.user_id} className="border-b hover:bg-gray-50">
                      <td className="p-3 text-gray-800">{child.email}</td>
                      <td className="p-3">
                        <Badge
                          className={
                            child.status === "active"
                              ? "bg-green-600"
                              : "bg-yellow-600"
                          }
                        >
                          {child.status}
                        </Badge>
                      </td>
                      <td className="p-3 text-gray-800 font-medium">
                        {child.total_points?.toFixed(2) || 0}
                      </td>
                      <td className="p-3 text-gray-600 text-sm">
                        {new Date(child.created_at).toLocaleDateString('fa-IR')}
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
