import { useState, useEffect } from "react";
import axios from "axios";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { TrendingUp, DollarSign, Users, Activity, RefreshCw, Loader2, Award, TreePine, Calendar } from "lucide-react";
import { toast } from "sonner";

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

export default function AdminReports() {
  const [activeTab, setActiveTab] = useState("financial");
  const [loading, setLoading] = useState(false);

  // Financial Reports
  const [financialSummary, setFinancialSummary] = useState(null);
  const [payments, setPayments] = useState([]);
  const [withdrawals, setWithdrawals] = useState([]);

  // Rewards Reports
  const [topEarners, setTopEarners] = useState([]);
  const [rewardsList, setRewardsList] = useState([]);
  const [rewardType, setRewardType] = useState("");

  // Users Reports
  const [topReferrers, setTopReferrers] = useState([]);
  const [treeStats, setTreeStats] = useState(null);
  const [activationTimeline, setActivationTimeline] = useState([]);
  const [timelineDays, setTimelineDays] = useState(30);

  // System Health
  const [systemHealth, setSystemHealth] = useState(null);

  // Pagination
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);

  const fetchFinancialSummary = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem("token");
      const response = await axios.get(`${API}/admin/reports/financial/summary`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setFinancialSummary(response.data);
    } catch (error) {
      toast.error("خطا در دریافت خلاصه مالی");
    } finally {
      setLoading(false);
    }
  };

  const fetchTopEarners = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem("token");
      const response = await axios.get(`${API}/admin/reports/rewards/top-earners?limit=20`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setTopEarners(response.data.top_earners);
    } catch (error) {
      toast.error("خطا در دریافت پردرآمدترین‌ها");
    } finally {
      setLoading(false);
    }
  };

  const fetchTopReferrers = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem("token");
      const response = await axios.get(`${API}/admin/reports/users/top-referrers?limit=20`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setTopReferrers(response.data.top_referrers);
    } catch (error) {
      toast.error("خطا در دریافت بیشترین دعوت‌کننده‌ها");
    } finally {
      setLoading(false);
    }
  };

  const fetchTreeStats = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem("token");
      const response = await axios.get(`${API}/admin/reports/users/tree-stats`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setTreeStats(response.data);
    } catch (error) {
      toast.error("خطا در دریافت آمار درخت");
    } finally {
      setLoading(false);
    }
  };

  const fetchSystemHealth = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem("token");
      const response = await axios.get(`${API}/admin/reports/system/health`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setSystemHealth(response.data);
    } catch (error) {
      toast.error("خطا در دریافت وضعیت سیستم");
    } finally {
      setLoading(false);
    }
  };

  const fetchActivationTimeline = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem("token");
      const response = await axios.get(
        `${API}/admin/reports/users/activation-timeline?days=${timelineDays}`,
        { headers: { Authorization: `Bearer ${token}` } }
      );
      setActivationTimeline(response.data.timeline);
    } catch (error) {
      toast.error("خطا در دریافت نمودار فعال‌سازی");
    } finally {
      setLoading(false);
    }
  };

  const fetchPayments = async (page = 1) => {
    setLoading(true);
    try {
      const token = localStorage.getItem("token");
      const response = await axios.get(
        `${API}/admin/reports/financial/payments?page=${page}&limit=20`,
        { headers: { Authorization: `Bearer ${token}` } }
      );
      setPayments(response.data.payments);
      setTotalPages(response.data.pages);
      setCurrentPage(page);
    } catch (error) {
      toast.error("خطا در دریافت پرداخت‌ها");
    } finally {
      setLoading(false);
    }
  };

  const fetchWithdrawals = async (page = 1) => {
    setLoading(true);
    try {
      const token = localStorage.getItem("token");
      const response = await axios.get(
        `${API}/admin/reports/financial/withdrawals?page=${page}&limit=20`,
        { headers: { Authorization: `Bearer ${token}` } }
      );
      setWithdrawals(response.data.withdrawals);
      setTotalPages(response.data.pages);
      setCurrentPage(page);
    } catch (error) {
      toast.error("خطا در دریافت برداشت‌ها");
    } finally {
      setLoading(false);
    }
  };

  const fetchRewards = async (page = 1) => {
    setLoading(true);
    try {
      const token = localStorage.getItem("token");
      let url = `${API}/admin/reports/rewards?page=${page}&limit=50`;
      if (rewardType) url += `&reward_type=${rewardType}`;
      
      const response = await axios.get(url, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setRewardsList(response.data.transactions);
      setTotalPages(response.data.pages);
      setCurrentPage(page);
    } catch (error) {
      toast.error("خطا در دریافت پاداش‌ها");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (activeTab === "financial") {
      fetchFinancialSummary();
      fetchPayments();
      fetchWithdrawals();
    } else if (activeTab === "rewards") {
      fetchTopEarners();
      fetchRewards();
    } else if (activeTab === "users") {
      fetchTopReferrers();
      fetchTreeStats();
      fetchActivationTimeline();
    } else if (activeTab === "system") {
      fetchSystemHealth();
    }
  }, [activeTab]);

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-bold text-gray-800">گزارشات سیستم</h2>
          <p className="text-sm text-gray-500">تحلیل کامل عملکرد سیستم</p>
        </div>
        <Button onClick={() => {
          if (activeTab === "financial") fetchFinancialSummary();
          else if (activeTab === "rewards") fetchTopEarners();
          else if (activeTab === "users") fetchTreeStats();
          else if (activeTab === "system") fetchSystemHealth();
        }} variant="outline" size="sm">
          <RefreshCw className="w-4 h-4 ml-2" />
          بروزرسانی
        </Button>
      </div>

      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList className="grid w-full grid-cols-4 gap-2">
          <TabsTrigger value="financial" className="flex items-center gap-2">
            <DollarSign className="w-4 h-4" />
            مالی
          </TabsTrigger>
          <TabsTrigger value="rewards" className="flex items-center gap-2">
            <Award className="w-4 h-4" />
            پاداش‌ها
          </TabsTrigger>
          <TabsTrigger value="users" className="flex items-center gap-2">
            <Users className="w-4 h-4" />
            کاربران
          </TabsTrigger>
          <TabsTrigger value="system" className="flex items-center gap-2">
            <Activity className="w-4 h-4" />
            سیستم
          </TabsTrigger>
        </TabsList>

        {/* Financial Tab */}
        <TabsContent value="financial" className="space-y-6">
          {loading ? (
            <div className="flex justify-center py-12">
              <Loader2 className="w-8 h-8 animate-spin text-blue-600" />
            </div>
          ) : financialSummary ? (
            <>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
                  <CardHeader>
                    <CardTitle className="text-lg">کل واریزها</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <p className="text-3xl font-bold text-green-700">
                      ${financialSummary.total_payments.toFixed(2)}
                    </p>
                  </CardContent>
                </Card>

                <Card className="bg-gradient-to-br from-red-50 to-red-100 border-red-200">
                  <CardHeader>
                    <CardTitle className="text-lg">کل برداشت‌ها</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <p className="text-3xl font-bold text-red-700">
                      ${financialSummary.total_withdrawals.toFixed(2)}
                    </p>
                  </CardContent>
                </Card>

                <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
                  <CardHeader>
                    <CardTitle className="text-lg">سود خالص</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <p className="text-3xl font-bold text-blue-700">
                      ${financialSummary.net_profit.toFixed(2)}
                    </p>
                  </CardContent>
                </Card>

                <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
                  <CardHeader>
                    <CardTitle className="text-lg">کل پاداش‌ها</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <p className="text-3xl font-bold text-purple-700">
                      ${financialSummary.total_rewards.toFixed(2)}
                    </p>
                  </CardContent>
                </Card>

                <Card className="bg-gradient-to-br from-yellow-50 to-yellow-100 border-yellow-200">
                  <CardHeader>
                    <CardTitle className="text-lg">موجودی کاربران</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <p className="text-3xl font-bold text-yellow-700">
                      ${financialSummary.total_user_balance.toFixed(2)}
                    </p>
                  </CardContent>
                </Card>

                <Card className="bg-gradient-to-br from-indigo-50 to-indigo-100 border-indigo-200">
                  <CardHeader>
                    <CardTitle className="text-lg">بالانس سیستم</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <p className="text-3xl font-bold text-indigo-700">
                      ${financialSummary.system_balance.toFixed(2)}
                    </p>
                  </CardContent>
                </Card>
              </div>
            </>
          ) : null}
        </TabsContent>

        {/* Rewards Tab */}
        <TabsContent value="rewards" className="space-y-6">
          {loading ? (
            <div className="flex justify-center py-12">
              <Loader2 className="w-8 h-8 animate-spin text-blue-600" />
            </div>
          ) : (
            <Card>
              <CardHeader>
                <CardTitle>پردرآمدترین کاربران</CardTitle>
                <CardDescription>20 کاربر برتر بر اساس امتیاز کل</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="overflow-x-auto">
                  <table className="w-full">
                    <thead>
                      <tr className="border-b">
                        <th className="text-right p-3">رتبه</th>
                        <th className="text-right p-3">ایمیل</th>
                        <th className="text-right p-3">امتیاز کل</th>
                        <th className="text-right p-3">عمق</th>
                      </tr>
                    </thead>
                    <tbody>
                      {topEarners.map((user, index) => (
                        <tr key={user.user_id} className="border-b hover:bg-gray-50">
                          <td className="p-3">
                            <Badge className={index < 3 ? "bg-yellow-600" : "bg-gray-600"}>
                              #{index + 1}
                            </Badge>
                          </td>
                          <td className="p-3">{user.email}</td>
                          <td className="p-3 font-bold text-green-600">
                            ${user.total_points.toFixed(2)}
                          </td>
                          <td className="p-3">{user.depth}</td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              </CardContent>
            </Card>
          )}
        </TabsContent>

        {/* Users Tab */}
        <TabsContent value="users" className="space-y-6">
          {loading ? (
            <div className="flex justify-center py-12">
              <Loader2 className="w-8 h-8 animate-spin text-blue-600" />
            </div>
          ) : (
            <>
              <Card>
                <CardHeader>
                  <CardTitle>بیشترین دعوت‌کننده‌ها</CardTitle>
                  <CardDescription>20 کاربر برتر بر اساس تعداد دعوت مستقیم</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="overflow-x-auto">
                    <table className="w-full">
                      <thead>
                        <tr className="border-b">
                          <th className="text-right p-3">رتبه</th>
                          <th className="text-right p-3">ایمیل</th>
                          <th className="text-right p-3">دعوت‌های مستقیم</th>
                          <th className="text-right p-3">امتیاز کل</th>
                        </tr>
                      </thead>
                      <tbody>
                        {topReferrers.map((user, index) => (
                          <tr key={user.user_id} className="border-b hover:bg-gray-50">
                            <td className="p-3">
                              <Badge className={index < 3 ? "bg-yellow-600" : "bg-gray-600"}>
                                #{index + 1}
                              </Badge>
                            </td>
                            <td className="p-3">{user.email}</td>
                            <td className="p-3 font-bold text-blue-600">
                              {user.direct_referrals}
                            </td>
                            <td className="p-3 text-green-600">
                              ${user.total_points.toFixed(2)}
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                </CardContent>
              </Card>

              {treeStats && (
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                      <TreePine className="w-5 h-5" />
                      آمار درخت
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-4">
                      <div className="flex items-center justify-between p-3 bg-blue-50 rounded-lg">
                        <span className="font-medium">کل کاربران فعال:</span>
                        <span className="text-2xl font-bold text-blue-700">
                          {treeStats.total_active_users}
                        </span>
                      </div>
                      
                      <div className="grid grid-cols-2 md:grid-cols-4 gap-3">
                        {treeStats.by_depth?.map((stat) => (
                          <div key={stat._id} className="bg-gray-50 p-3 rounded-lg text-center">
                            <p className="text-sm text-gray-600">عمق {stat._id}</p>
                            <p className="text-xl font-bold text-gray-800">{stat.count}</p>
                            <p className="text-xs text-gray-500">
                              میانگین: {stat.avg_children?.toFixed(1)}
                            </p>
                          </div>
                        ))}
                      </div>
                    </div>
                  </CardContent>
                </Card>
              )}
            </>
          )}
        </TabsContent>

        {/* System Tab */}
        <TabsContent value="system" className="space-y-6">
          {loading ? (
            <div className="flex justify-center py-12">
              <Loader2 className="w-8 h-8 animate-spin text-blue-600" />
            </div>
          ) : systemHealth ? (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">کاربران</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-2">
                    <div className="flex justify-between">
                      <span>کل:</span>
                      <span className="font-bold">{systemHealth.users.total}</span>
                    </div>
                    <div className="flex justify-between">
                      <span>فعال:</span>
                      <span className="font-bold text-green-600">{systemHealth.users.active}</span>
                    </div>
                    <div className="flex justify-between">
                      <span>در انتظار:</span>
                      <span className="font-bold text-yellow-600">{systemHealth.users.pending}</span>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">پرداخت‌ها</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-2">
                    <div className="flex justify-between">
                      <span>در انتظار:</span>
                      <span className="font-bold text-yellow-600">{systemHealth.payments.pending}</span>
                    </div>
                    <div className="flex justify-between">
                      <span>تأیید شده:</span>
                      <span className="font-bold text-green-600">{systemHealth.payments.confirmed}</span>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">برداشت‌ها</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-2">
                    <div className="flex justify-between">
                      <span>در انتظار:</span>
                      <span className="font-bold text-yellow-600">{systemHealth.withdrawals.pending}</span>
                    </div>
                    <div className="flex justify-between">
                      <span>تکمیل شده:</span>
                      <span className="font-bold text-green-600">{systemHealth.withdrawals.completed}</span>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="md:col-span-2">
                <CardHeader>
                  <CardTitle className="text-lg">مالی</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <p className="text-sm text-gray-600">کل پاداش‌ها:</p>
                      <p className="text-2xl font-bold text-purple-600">
                        ${systemHealth.financial.total_rewards_paid.toFixed(2)}
                      </p>
                    </div>
                    <div>
                      <p className="text-sm text-gray-600">موجودی کاربران:</p>
                      <p className="text-2xl font-bold text-blue-600">
                        ${systemHealth.financial.total_user_balance.toFixed(2)}
                      </p>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">تراکنش‌ها</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="flex justify-between items-center">
                    <span>کل تراکنش‌ها:</span>
                    <span className="text-2xl font-bold text-indigo-600">
                      {systemHealth.transactions.total}
                    </span>
                  </div>
                </CardContent>
              </Card>
            </div>
          ) : null}
        </TabsContent>
      </Tabs>
    </div>
  );
}
