import { useState, useEffect } from "react";
import "@/App.css";
import { BrowserRouter, Routes, Route, Navigate, useNavigate } from "react-router-dom";
import axios from "axios";
import Login from "@/components/Login";
import Register from "@/components/Register";
import UserDashboard from "@/components/UserDashboard";
import AdminDashboard from "@/components/AdminDashboard";
import { Toaster } from "@/components/ui/sonner";
import ErrorBoundary from "@/components/ErrorBoundary";

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL || '';
const API = `${BACKEND_URL}/api`;

console.log("App loaded - API URL:", API);

function App() {
  const [user, setUser] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const token = localStorage.getItem("token");
    console.log("useEffect - token exists:", !!token);
    
    if (token) {
      axios.defaults.headers.common["Authorization"] = `Bearer ${token}`;
      fetchProfile();
    } else {
      setLoading(false);
    }
  }, []);

  const fetchProfile = async () => {
    console.log("fetchProfile - calling:", `${API}/user/profile`);
    try {
      const response = await axios.get(`${API}/user/profile`);
      console.log("Profile success:", response.data);
      setUser(response.data);
    } catch (e) {
      console.error("Profile error:", e.response?.status, e.message);
      localStorage.removeItem("token");
      delete axios.defaults.headers.common["Authorization"];
      setUser(null);
    } finally {
      setLoading(false);
    }
  };

  const handleLogin = (token, userData) => {
    console.log("handleLogin - userData:", userData);
    localStorage.setItem("token", token);
    axios.defaults.headers.common["Authorization"] = `Bearer ${token}`;
    
    // Set user و loading false
    setUser(userData);
    setLoading(false);
    
    // Navigation بدون reload - فقط با React state
    // حذف window.location.href
  };

  const handleLogout = () => {
    localStorage.removeItem("token");
    delete axios.defaults.headers.common["Authorization"];
    setUser(null);
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-emerald-50 to-teal-100">
        <div className="text-center">
          <div className="animate-spin rounded-full h-16 w-16 border-b-2 border-emerald-600 mx-auto"></div>
          <p className="mt-4 text-emerald-700 font-vazir">در حال بارگذاری...</p>
        </div>
      </div>
    );
  }

  return (
    <ErrorBoundary>
      <div className="App" dir="rtl">
        <Toaster position="top-center" richColors />
        <BrowserRouter>
          <Routes>
            <Route
              path="/login"
              element={
                !user ? <Login onLogin={handleLogin} /> : <Navigate to="/" />
              }
            />
            <Route
              path="/register"
              element={
                !user ? <Register onLogin={handleLogin} /> : <Navigate to="/" />
              }
            />
            <Route
              path="/"
              element={
                user ? (
                  user.role === "admin" ? (
                    <AdminDashboard user={user} onLogout={handleLogout} />
                  ) : (
                    <UserDashboard user={user} onLogout={handleLogout} />
                  )
                ) : (
                  <Navigate to="/login" />
                )
              }
            />
          </Routes>
        </BrowserRouter>
      </div>
    </ErrorBoundary>
  );
}

export default App;
