"""Crypto Routes - اندپوینت‌های کریپتو"""

import uuid
import logging
from datetime import datetime, timezone, timedelta
from typing import Optional

from fastapi import APIRouter, HTTPException, Depends, Query

from models.payment import CryptoPaymentCreate, CryptoSettings
from models.withdrawal import WithdrawalRequest
from services.auth_service import get_current_user, get_admin_user
from services.tron_service import tron_service
from services.bsc_service import bsc_service
from utils.validators import validate_wallet_address
from utils.helpers import generate_unique_amount
from database import get_db

router = APIRouter()


@router.post("/user/crypto/payment")
async def create_crypto_payment(data: CryptoPaymentCreate, current_user: dict = Depends(get_current_user)):
    """ایجاد پرداخت کریپتو"""
    db = await get_db()
    
    # دریافت تنظیمات
    crypto_settings = await db.settings.find_one({"setting_id": "crypto"}, {"_id": 0})
    reward_settings = await db.settings.find_one({"setting_id": "rewards"}, {"_id": 0})
    
    if not crypto_settings:
        raise HTTPException(status_code=400, detail="تنظیمات کریپتو تنظیم نشده")
    
    network = data.network.upper()
    
    # بررسی فعال بودن شبکه
    if network == "TRON" and not crypto_settings.get("tron_enabled"):
        raise HTTPException(status_code=400, detail="شبکه TRON غیرفعال است")
    if network == "BSC" and not crypto_settings.get("bsc_enabled"):
        raise HTTPException(status_code=400, detail="شبکه BSC غیرفعال است")
    
    # دریافت آدرس کیف پول
    wallet_address = crypto_settings.get(f"{network.lower()}_wallet_address")
    if not wallet_address:
        raise HTTPException(status_code=400, detail=f"آدرس کیف پول {network} تنظیم نشده")
    
    # ایجاد payment
    payment_id = str(uuid.uuid4())
    base_amount = reward_settings.get("activation_fee", 100.0) if reward_settings else 100.0
    unique_amount = generate_unique_amount(base_amount, payment_id)
    
    payment = {
        "payment_id": payment_id,
        "user_id": current_user["user_id"],
        "amount": base_amount,
        "unique_amount": unique_amount,
        "network": network,
        "wallet_address": wallet_address,
        "status": "pending",
        "tx_hash": None,
        "created_at": datetime.now(timezone.utc).isoformat(),
        "confirmed_at": None,
        "expires_at": (datetime.now(timezone.utc) + timedelta(hours=24)).isoformat()
    }
    
    await db.crypto_payments.insert_one(payment)
    
    return {
        "payment_id": payment_id,
        "amount": unique_amount,
        "wallet_address": wallet_address,
        "network": network,
        "expires_at": payment["expires_at"]
    }


@router.get("/user/crypto/payment/{payment_id}")
async def get_payment_status(payment_id: str, current_user: dict = Depends(get_current_user)):
    """وضعیت پرداخت"""
    db = await get_db()
    
    payment = await db.crypto_payments.find_one(
        {"payment_id": payment_id, "user_id": current_user["user_id"]},
        {"_id": 0}
    )
    
    if not payment:
        raise HTTPException(status_code=404, detail="پرداخت یافت نشد")
    
    return payment


@router.post("/user/withdrawal/request")
async def request_withdrawal(data: WithdrawalRequest, current_user: dict = Depends(get_current_user)):
    """درخواست برداشت"""
    db = await get_db()
    
    # بررسی موجودی
    if current_user.get("total_points", 0) < data.amount:
        raise HTTPException(status_code=400, detail="موجودی کافی نیست")
    
    # بررسی آدرس
    if not validate_wallet_address(data.wallet_address, data.network):
        raise HTTPException(status_code=400, detail="آدرس کیف پول نامعتبر")
    
    # دریافت تنظیمات
    crypto_settings = await db.settings.find_one({"setting_id": "crypto"}, {"_id": 0})
    min_amount = crypto_settings.get("min_withdrawal_amount", 10) if crypto_settings else 10
    
    if data.amount < min_amount:
        raise HTTPException(status_code=400, detail=f"حداقل مبلغ برداشت {min_amount} است")
    
    # محاسبه کارمزد
    fee = 0.0
    if crypto_settings:
        fee_type = crypto_settings.get("withdrawal_fee_type", "fixed")
        fee_amount = crypto_settings.get("withdrawal_fee_amount", 0)
        
        if fee_type == "fixed":
            fee = fee_amount
        else:
            fee = data.amount * (fee_amount / 100)
    
    net_amount = data.amount - fee
    
    # ایجاد درخواست
    withdrawal = {
        "withdrawal_id": str(uuid.uuid4()),
        "user_id": current_user["user_id"],
        "amount": data.amount,
        "fee": fee,
        "net_amount": net_amount,
        "wallet_address": data.wallet_address,
        "network": data.network.upper(),
        "status": "pending_admin",
        "tx_hash": None,
        "rejection_reason": None,
        "requested_at": datetime.now(timezone.utc).isoformat(),
        "processed_at": None,
        "processed_by": None
    }
    
    await db.withdrawals.insert_one(withdrawal)
    
    # کسر از موجودی
    await db.users.update_one(
        {"user_id": current_user["user_id"]},
        {"$inc": {"total_points": -data.amount}}
    )
    
    logging.info(f"✅ Withdrawal request created: {withdrawal['withdrawal_id']}")
    
    return {
        "withdrawal_id": withdrawal["withdrawal_id"],
        "amount": data.amount,
        "fee": fee,
        "net_amount": net_amount,
        "status": "pending_admin"
    }


@router.get("/user/withdrawals")
async def get_user_withdrawals(current_user: dict = Depends(get_current_user)):
    """لیست درخواست‌های برداشت کاربر"""
    db = await get_db()
    
    withdrawals = await db.withdrawals.find(
        {"user_id": current_user["user_id"]},
        {"_id": 0}
    ).sort("requested_at", -1).limit(50).to_list(50)
    
    return withdrawals


# Admin Routes

@router.get("/admin/withdrawals")
async def get_all_withdrawals(
    status: Optional[str] = None,
    page: int = Query(1, ge=1),
    limit: int = Query(20, ge=1, le=100),
    admin_user: dict = Depends(get_admin_user)
):
    """لیست همه درخواست‌های برداشت"""
    db = await get_db()
    
    filter_query = {}
    if status:
        filter_query["status"] = status
    
    total = await db.withdrawals.count_documents(filter_query)
    skip = (page - 1) * limit
    
    withdrawals = await db.withdrawals.find(
        filter_query,
        {"_id": 0}
    ).sort("requested_at", -1).skip(skip).limit(limit).to_list(limit)
    
    return {
        "withdrawals": withdrawals,
        "total": total,
        "page": page,
        "pages": (total + limit - 1) // limit
    }


@router.put("/admin/withdrawals/{withdrawal_id}/approve")
async def approve_withdrawal(withdrawal_id: str, tx_hash: str = None, admin_user: dict = Depends(get_admin_user)):
    """تایید درخواست برداشت"""
    db = await get_db()
    
    await db.withdrawals.update_one(
        {"withdrawal_id": withdrawal_id},
        {"$set": {
            "status": "completed",
            "tx_hash": tx_hash,
            "processed_at": datetime.now(timezone.utc).isoformat(),
            "processed_by": admin_user["user_id"]
        }}
    )
    
    return {"message": "درخواست تایید شد"}


@router.put("/admin/withdrawals/{withdrawal_id}/reject")
async def reject_withdrawal(withdrawal_id: str, reason: str = None, admin_user: dict = Depends(get_admin_user)):
    """رد درخواست برداشت"""
    db = await get_db()
    
    withdrawal = await db.withdrawals.find_one({"withdrawal_id": withdrawal_id})
    if not withdrawal:
        raise HTTPException(status_code=404, detail="درخواست یافت نشد")
    
    # برگشت مبلغ
    await db.users.update_one(
        {"user_id": withdrawal["user_id"]},
        {"$inc": {"total_points": withdrawal["amount"]}}
    )
    
    await db.withdrawals.update_one(
        {"withdrawal_id": withdrawal_id},
        {"$set": {
            "status": "rejected",
            "rejection_reason": reason,
            "processed_at": datetime.now(timezone.utc).isoformat(),
            "processed_by": admin_user["user_id"]
        }}
    )
    
    return {"message": "درخواست رد شد"}


@router.get("/admin/settings/crypto")
async def get_crypto_settings(admin_user: dict = Depends(get_admin_user)):
    """دریافت تنظیمات کریپتو"""
    db = await get_db()
    settings = await db.settings.find_one({"setting_id": "crypto"}, {"_id": 0})
    return settings or {}


@router.put("/admin/settings/crypto")
async def update_crypto_settings(data: CryptoSettings, admin_user: dict = Depends(get_admin_user)):
    """بروزرسانی تنظیمات کریپتو"""
    db = await get_db()
    
    settings_dict = data.model_dump()
    settings_dict["setting_id"] = "crypto"
    
    await db.settings.update_one(
        {"setting_id": "crypto"},
        {"$set": settings_dict},
        upsert=True
    )
    
    # تنظیم API key برای سرویس‌ها
    if data.tron_api_key:
        tron_service.set_api_key(data.tron_api_key)
    
    return {"message": "تنظیمات کریپتو بروزرسانی شد"}
