# 📘 Developer Guide - MLM با Materialized Path

## 🎯 هدف این راهنما

این راهنما برای توسعه‌دهندگانی است که می‌خواهند:
- سیستم را توسعه دهند
- منطق جایزه را تغییر دهند
- ساختار درخت را تغییر دهند
- feature جدید اضافه کنند

---

## 📁 ساختار پروژه و مسئولیت‌ها

```
backend/
├── services/
│   ├── tree/              → 🌳 همه چیز مربوط به درخت
│   ├── rewards/           → 💰 همه چیز مربوط به پاداش
│   └── user/              → 👤 همه چیز مربوط به کاربر
```

### 🌳 ماژول Tree (`services/tree/`)

**مسئولیت:** مدیریت ساختار درخت با Materialized Path

| فایل | مسئولیت | تغییرات معمول |
|------|---------|--------------|
| `path_manager.py` | محاسبات path | **تغییر ندهید** - منطق ثابت |
| `placement.py` | جایگذاری کاربر | اگر می‌خواهید الگوریتم جایگذاری تغییر کند |
| `tree_stats.py` | آمار درخت | اگر می‌خواهید روش محاسبه آمار تغییر کند |

**مثال: تغییر الگوریتم جایگذاری**

اگر می‌خواهید به جای "کمترین depth" روش دیگری باشد:

```python
# services/tree/placement.py
# خط ~58

# قبل:
parent = await _db.users.find_one(
    {...},
    sort=[("depth", 1)]  # کمترین depth اول
)

# بعد (مثلاً: کمترین فرزند اول):
parent = await _db.users.find_one(
    {...},
    sort=[("direct_children_count", 1)]  # کمترین فرزند اول
)
```

---

### 💰 ماژول Rewards (`services/rewards/`)

**مسئولیت:** محاسبه و توزیع پاداش

| فایل | مسئولیت | تغییرات معمول |
|------|---------|--------------|
| `calculator.py` | محاسبه مبالغ پاداش | ✅ **اینجا تغییر دهید** برای تغییر منطق پاداش |
| `distributor.py` | توزیع پاداش‌ها | اگر می‌خواهید روش توزیع تغییر کند |
| `validator.py` | قوانین پاداش | اگر می‌خواهید قوانین جدید اضافه کنید |

**مثال 1: تغییر مبالغ پاداش مستقیم**

```python
# services/rewards/calculator.py
# خط ~68-78

# اگر می‌خواهید مبلغ نفر دوم 80 دلار باشد:
elif active_children_count == 1:
    amount = settings.get("direct_second", 80.0)  # قبلاً 75.0 بود
    desc = "مستقیم نفر دوم"
```

**مثال 2: اضافه کردن Level 9 reward**

```python
# services/rewards/calculator.py
# در تابع calculate_level_rewards
# خط ~125

# اضافه کردن:
elif depth_diff == 8:  # Level 9
    reward_amount = settings.get("level_9", 1.0)
    reward_type = "level_9"
    level_name = "سطح ۹"
```

**⚠️ نکته مهم:** اگر level جدید اضافه می‌کنید:
1. به `database` در collection `settings` فیلد `level_9` اضافه کنید
2. به `models/reward.py` فیلد `level_9` اضافه کنید

---

### 👤 ماژول User (`services/user/`)

**مسئولیت:** مدیریت کاربران (ثبت‌نام، فعال‌سازی، پروفایل)

| فایل | مسئولیت | تغییرات معمول |
|------|---------|--------------|
| `registration.py` | ثبت‌نام کاربر | اگر می‌خواهید فیلد جدید اضافه کنید |
| `activation.py` | فعال‌سازی کاربر | **کمتر تغییر دهید** - منطق پیچیده |
| `profile.py` | پروفایل کاربر | برای feature‌های پروفایل |

**مثال: اضافه کردن فیلد جدید در ثبت‌نام**

```python
# services/user/registration.py
# خط ~80

new_user = {
    ...
    "phone_number": phone_number,
    "country": country,  # فیلد جدید
    ...
}
```

---

## 🔧 سناریوهای معمول تغییرات

### 📌 سناریو 1: تغییر منطق پاداش مستقیم

**هدف:** می‌خواهم پاداش مستقیم بر اساس تعداد کل زیرمجموعه‌ها باشد، نه ترتیب

**فایل:** `services/rewards/calculator.py`
**تابع:** `calculate_direct_reward()`

```python
# خط ~68

# قبل:
active_children_count = await _db.users.count_documents({
    "referrer_id": referrer_id,
    "status": "active",
    "user_id": {"$ne": new_user_id}
})

if active_children_count == 0:
    amount = settings.get("direct_first", 100.0)
elif active_children_count == 1:
    amount = settings.get("direct_second", 75.0)
...

# بعد (بر اساس کل):
total_descendants = referrer.get("total_descendants", 0)

if total_descendants < 10:
    amount = 100.0
elif total_descendants < 50:
    amount = 75.0
else:
    amount = 50.0
```

---

### 📌 سناریو 2: تغییر تعداد فرزند مجاز

**هدف:** می‌خواهم هر کاربر 5 فرزند داشته باشد، نه 3

**تغییرات لازم:**

1. **Database Model**: `models/user.py`
```python
# محدودیت در comment تغییر کند
direct_children: List[str] = []  # max 5
```

2. **Placement Logic**: `services/tree/placement.py`
```python
# خط ~39 و ~65
"direct_children_count": {"$lt": 5}  # قبلاً 3 بود
```

3. **Tree Completion**: `services/tree/tree_stats.py`
```python
# خط ~81
# محاسبه جدید برای هرم 5-ary:
# سطح 1: 5
# سطح 2: 25
# سطح 3: 125
# ...
# جمع 7 سطح: 5 + 25 + 125 + 625 + 3125 + 15625 + 78125 = 97655
MAX_DESCENDANTS = 97655  # قبلاً 3279 بود
```

---

### 📌 سناریو 3: اضافه کردن Level جدید

**هدف:** اضافه کردن Level 4 با depth_diff=3 و مبلغ $8

**تغییرات لازم:**

1. **Settings**: در MongoDB collection `settings`
```javascript
db.settings.updateOne(
  {"setting_id": "rewards"},
  {"$set": {"level_4": 8.0}}
)
```

2. **Model**: `models/reward.py`
```python
level_4: float = Field(default=8.0, ge=0, description="پاداش سطح 4")
```

3. **Calculator**: `services/rewards/calculator.py`
```python
# خط ~116 - اضافه کردن elif جدید
elif depth_diff == 3:
    reward_amount = settings.get("level_4", 8.0)
    reward_type = "level_4"
    level_name = "سطح ۴"
```

4. **Validator**: `services/rewards/validator.py`
```python
# خط ~45
return depth_diff in [2, 3, 4, 6]  # 3 اضافه شد
```

---

### 📌 سناریو 4: تغییر الگوریتم جایگذاری به "بهترین parent"

**هدف:** به جای اولین parent خالی، parent با بیشترین امتیاز را انتخاب کن

**فایل:** `services/tree/placement.py`
**تابع:** `find_available_parent()`

```python
# خط ~64

# قبل:
parent = await _db.users.find_one(
    {
        "status": "active",
        "direct_children_count": {"$lt": 3}
    },
    sort=[("depth", 1)]  # کمترین depth
)

# بعد:
parent = await _db.users.find_one(
    {
        "status": "active",
        "direct_children_count": {"$lt": 3}
    },
    sort=[
        ("total_points", -1),  # بیشترین امتیاز اول
        ("depth", 1)           # اگر امتیاز برابر، کمترین depth
    ]
)
```

---

## ⚠️ نقاط حساس - نباید تغییر داده شوند

### 🔴 خطر بالا

**1. Path Manager (`services/tree/path_manager.py`)**
- **هرگز** فرمت path را تغییر ندهید
- **هرگز** جداکننده `.` را تغییر ندهید
- این کد منطق ثابت دارد و تغییر آن کل سیستم را خراب می‌کند

**2. Ancestor Depths Structure**
```python
# این ساختار ثابت است:
ancestor_depths = [
    {"user_id": "...", "depth": 0},
    {"user_id": "...", "depth": 1}
]
# تغییر این ساختار باعث خرابی reward calculation می‌شود
```

**3. Activation Flow (`services/user/activation.py`)**
- این فایل منطق پیچیده دارد
- **قبل از تغییر** حتماً با تیم مشورت کنید
- تغییر ترتیب عملیات باعث inconsistency می‌شود

---

### 🟡 خطر متوسط

**1. Distributor (`services/rewards/distributor.py`)**
- Bulk operations را تغییر ندهید
- فقط منطق محاسبه را تغییر دهید

**2. Tree Stats (`services/tree/tree_stats.py`)**
- regex pattern در query را دقیق تغییر دهید
- عدد MAX_DESCENDANTS را با دقت محاسبه کنید

---

## 🧪 تست کردن تغییرات

### تست Local

```bash
# 1. تست ثبت‌نام
curl -X POST http://localhost:8001/api/auth/register \
  -H "Content-Type: application/json" \
  -d '{"email":"test@test.com","password":"123456"}'

# 2. تست فعال‌سازی (نیاز به token ادمین)
curl -X POST http://localhost:8001/api/admin/users/{user_id}/activate \
  -H "Authorization: Bearer {admin_token}"

# 3. چک پاداش‌ها
curl http://localhost:8001/api/user/transactions \
  -H "Authorization: Bearer {user_token}"
```

### تست Performance

```python
# test_performance.py
import asyncio
import time

async def test_activation_speed():
    start = time.time()
    
    # فعال‌سازی 100 کاربر
    for i in range(100):
        await activate_user(f"user_{i}")
    
    duration = time.time() - start
    print(f"100 users in {duration}s")
    print(f"Average: {duration/100}s per user")
    
    # انتظار: < 0.5s per user
    assert duration < 50, "Too slow!"

asyncio.run(test_activation_speed())
```

---

## 📊 Monitoring و Debugging

### لاگ‌های مهم

```python
# در هر سرویس logger وجود دارد
logger.info(f"✅ Success message")
logger.warning(f"⚠️ Warning message")
logger.error(f"❌ Error message")
```

**مکان لاگ‌ها:**
```bash
tail -f /var/log/supervisor/backend.out.log
tail -f /var/log/supervisor/backend.err.log
```

### Query Monitoring

```python
# اضافه کردن logging برای queries
import time

async def logged_query(query_name, coro):
    start = time.time()
    result = await coro
    duration = time.time() - start
    logger.info(f"Query [{query_name}]: {duration}s")
    return result

# استفاده:
users = await logged_query(
    "find_users",
    _db.users.find({...}).to_list(100)
)
```

---

## 🚀 بهترین روش‌ها (Best Practices)

### 1. همیشه Index بررسی کنید

```python
# قبل از query جدید:
# آیا index مناسب وجود دارد؟
await _db.users.create_index("new_field")
```

### 2. از Bulk Operations استفاده کنید

```python
# ❌ بد:
for user in users:
    await _db.users.update_one({"user_id": user}, {...})

# ✅ خوب:
updates = [UpdateOne({"user_id": u}, {...}) for u in users]
await _db.users.bulk_write(updates)
```

### 3. Error Handling

```python
# همیشه try-except
try:
    result = await some_operation()
except Exception as e:
    logger.error(f"❌ Operation failed: {e}")
    # اقدام جبرانی
```

### 4. Transaction برای عملیات مهم

```python
async with await client.start_session() as session:
    async with session.start_transaction():
        await _db.users.update_one({...}, session=session)
        await _db.payments.insert_one({...}, session=session)
        # اگر خطا: rollback خودکار
```

---

## 📚 منابع بیشتر

- **طراحی کامل:** `/MATERIALIZED_PATH_DESIGN.md`
- **تحلیل سیستم قبلی:** `/MLM_SYSTEM_ANALYSIS.md`
- **MongoDB Docs:** https://docs.mongodb.com/
- **FastAPI Docs:** https://fastapi.tiangolo.com/

---

## ❓ سوالات متداول

**Q: چرا از Materialized Path استفاده می‌کنیم؟**
A: برای حذف کامل traversal و آماده‌سازی برای 100K+ users

**Q: آیا می‌توانم path را به format دیگری تغییر دهم؟**
A: خیر - تغییر path format نیاز به migration کامل دارد

**Q: چگونه می‌توانم query count را بررسی کنم؟**
A: از MongoDB profiler استفاده کنید یا logging اضافه کنید

**Q: اگر تغییرات من سیستم را خراب کرد چه کنم؟**
A: از Git استفاده کنید:
```bash
git checkout -- .
git clean -fd
```

---

**نسخه:** 1.0
**تاریخ:** 2025-12-17
**مخاطب:** توسعه‌دهندگان Backend
