import { useState, useEffect } from "react";
import axios from "axios";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import { toast } from "sonner";
import { LogOut, Copy, Users, TrendingUp, Share2, Network, Award, Clock, CheckCircle, XCircle, DollarSign, Wallet, MessageSquare, Settings, BookOpen, GraduationCap, MessageCircle } from "lucide-react";
import CryptoPaymentPage from "@/components/CryptoPaymentPage";
import WithdrawalPage from "@/components/WithdrawalPage";
import MessagingSystem from "@/components/MessagingSystem";
import UserSettings from "@/components/UserSettings";
import AnnouncementSlider from "@/components/AnnouncementSlider";
import AnnouncementModal from "@/components/AnnouncementModal";
import IframePage from "@/components/IframePage";
import TelegramSupport from "@/components/TelegramSupport";

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

function UserDashboard({ user, onLogout }) {
  const [profile, setProfile] = useState(null);
  const [transactions, setTransactions] = useState([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchData();
  }, []);

  const fetchData = async () => {
    try {
      const [profileRes, transactionsRes] = await Promise.all([
        axios.get(`${API}/user/profile`),
        axios.get(`${API}/user/transactions`),
      ]);

      setProfile(profileRes.data);
      setTransactions(transactionsRes.data);
    } catch (error) {
      toast.error("خطا در دریافت اطلاعات");
    } finally {
      setLoading(false);
    }
  };

  const copyToClipboard = (text, successMessage) => {
    try {
      if (navigator.clipboard && navigator.clipboard.writeText) {
        navigator.clipboard.writeText(text).then(() => {
          toast.success(successMessage);
        }).catch(() => {
          // Fallback به روش قدیمی
          const textarea = document.createElement('textarea');
          textarea.value = text;
          textarea.style.position = 'fixed';
          textarea.style.opacity = '0';
          document.body.appendChild(textarea);
          textarea.select();
          document.execCommand('copy');
          document.body.removeChild(textarea);
          toast.success(successMessage);
        });
      } else {
        // Fallback به روش قدیمی
        const textarea = document.createElement('textarea');
        textarea.value = text;
        textarea.style.position = 'fixed';
        textarea.style.opacity = '0';
        document.body.appendChild(textarea);
        textarea.select();
        document.execCommand('copy');
        document.body.removeChild(textarea);
        toast.success(successMessage);
      }
    } catch (err) {
      toast.error("خطا در کپی - لطفاً دستی کپی کنید");
    }
  };

  const copyReferralLink = () => {
    const link = `${window.location.origin}/register?ref=${user.referral_code}`;
    copyToClipboard(link, "لینک دعوت کپی شد");
  };

  const [telegramBotUsername, setTelegramBotUsername] = useState("YourBotUsername");

  // دریافت bot username از تنظیمات
  useEffect(() => {
    const fetchTelegramSettings = async () => {
      try {
        const response = await axios.get(`${API}/user/telegram/bot-info`);
        if (response.data && response.data.bot_username) {
          setTelegramBotUsername(response.data.bot_username);
        }
      } catch (error) {
        console.error('Error fetching telegram settings:', error);
      }
    };
    fetchTelegramSettings();
  }, []);

  const copyTelegramLink = () => {
    const telegramLink = `https://t.me/${telegramBotUsername}?start=${user.referral_code}`;
    copyToClipboard(telegramLink, "لینک تلگرام کپی شد");
  };

  const getStatusBadge = (status) => {
    const statusMap = {
      pending: { label: "در انتظار", variant: "outline", icon: Clock },
      active: { label: "فعال", variant: "default", icon: CheckCircle },
      exited: { label: "خارج شده", variant: "destructive", icon: XCircle },
    };
    const statusInfo = statusMap[status] || statusMap.pending;
    const Icon = statusInfo.icon;
    return (
      <Badge variant={statusInfo.variant} className="flex items-center gap-1">
        <Icon className="w-3 h-3" />
        {statusInfo.label}
      </Badge>
    );
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-emerald-50 to-teal-100">
        <div className="text-center">
          <div className="animate-spin rounded-full h-16 w-16 border-b-2 border-emerald-600 mx-auto"></div>
          <p className="mt-4 text-emerald-700 font-vazir">در حال بارگذاری...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-teal-50 to-cyan-50" data-testid="user-dashboard">
      {/* Announcement Modal */}
      <AnnouncementModal />
      
      {/* Header */}
      <div className="bg-white shadow-sm border-b">
        <div className="container mx-auto px-4 py-4 flex justify-between items-center">
          <div className="flex items-center gap-3">
            <div className="w-12 h-12 bg-gradient-to-br from-emerald-500 to-teal-600 rounded-xl flex items-center justify-center shadow-md">
              <Users className="w-6 h-6 text-white" />
            </div>
            <div>
              <h1 className="text-2xl font-bold text-emerald-900">داشبورد کاربر</h1>
              <p className="text-sm text-gray-600">{profile?.email}</p>
            </div>
          </div>
          <Button
            onClick={onLogout}
            variant="outline"
            className="flex items-center gap-2 border-red-200 text-red-600 hover:bg-red-50"
            data-testid="logout-button"
          >
            <LogOut className="w-4 h-4" />
            خروج
          </Button>
        </div>
      </div>

      {/* Main Content */}
      <div className="container mx-auto px-4 py-8">
        {/* Status Cards */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-6 mb-8">
          <Card className="bg-gradient-to-br from-emerald-500 to-teal-600 text-white border-0 shadow-xl" data-testid="status-card">
            <CardHeader className="pb-3">
              <CardDescription className="text-emerald-100">وضعیت حساب</CardDescription>
              <CardTitle className="text-3xl flex items-center gap-2">
                {getStatusBadge(profile?.status)}
              </CardTitle>
            </CardHeader>
          </Card>

          <Card className="bg-white shadow-lg border-0" data-testid="points-card">
            <CardHeader className="pb-3">
              <CardDescription className="text-gray-600 flex items-center gap-2">
                <Award className="w-4 h-4" />
                مجموع امتیاز
              </CardDescription>
              <CardTitle className="text-3xl text-emerald-600">
                {profile?.total_points?.toFixed(0) || '0'}
              </CardTitle>
            </CardHeader>
          </Card>

          <Card className="bg-white shadow-lg border-0" data-testid="direct-referrals-card">
            <CardHeader className="pb-3">
              <CardDescription className="text-gray-600 flex items-center gap-2">
                <Users className="w-4 h-4" />
                دعوت مستقیم (فعال)
              </CardDescription>
              <CardTitle className="text-3xl text-teal-600">
                {profile?.direct_referrals - (profile?.pending_referrals || 0)}
              </CardTitle>
            </CardHeader>
          </Card>

          <Card className="bg-white shadow-lg border-0" data-testid="pending-referrals-card">
            <CardHeader className="pb-3">
              <CardDescription className="text-gray-600 flex items-center gap-2">
                <Clock className="w-4 h-4" />
                در انتظار فعال‌سازی
              </CardDescription>
              <CardTitle className="text-3xl text-yellow-600">
                {profile?.pending_referrals || 0}
              </CardTitle>
            </CardHeader>
          </Card>

          {/* Announcement Slider */}
          <div className="col-span-full">
            <AnnouncementSlider />
          </div>
        </div>

        {/* Referral Links */}
        <Card className="mb-8 shadow-lg border-0" data-testid="referral-link-card">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Share2 className="w-5 h-5 text-emerald-600" />
              لینک‌های دعوت شما
            </CardTitle>
            <CardDescription>
              {user.status === 'active' 
                ? 'با اشتراک‌گذاری این لینک‌ها، دوستان خود را دعوت کنید'
                : 'لینک‌های دعوت بعد از فعالسازی حساب در دسترس خواهد بود'}
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            {user.status === 'active' ? (
              <>
                {/* کد معرف */}
                <div className="p-4 bg-gradient-to-r from-emerald-50 to-teal-50 rounded-xl border-2 border-emerald-200">
                  <p className="text-sm text-gray-600 mb-2">🎯 کد معرف شما:</p>
                  <p className="font-mono text-2xl font-bold text-emerald-700 text-center" data-testid="referral-code">
                    {user.referral_code}
                  </p>
                </div>

                {/* لینک عادی */}
                <div className="space-y-2">
                  <div className="flex items-center justify-between">
                    <p className="text-sm font-semibold text-gray-700">🌐 لینک دعوت (وب):</p>
                    <Button
                      onClick={copyReferralLink}
                      size="sm"
                      className="bg-gradient-to-r from-emerald-600 to-teal-600 hover:from-emerald-700 hover:to-teal-700"
                      data-testid="copy-referral-link-button"
                    >
                      <Copy className="w-4 h-4 ml-1" />
                      کپی
                    </Button>
                  </div>
                  <div className="p-3 bg-white rounded-lg border border-gray-200 text-xs break-all">
                    {`${window.location.origin}/register?ref=${user.referral_code}`}
                  </div>
                </div>

                {/* لینک تلگرام */}
                <div className="space-y-2">
                  <div className="flex items-center justify-between">
                    <p className="text-sm font-semibold text-gray-700">✈️ لینک دعوت (تلگرام):</p>
                    <Button
                      onClick={copyTelegramLink}
                      size="sm"
                      className="bg-gradient-to-r from-blue-600 to-cyan-600 hover:from-blue-700 hover:to-cyan-700"
                      data-testid="copy-telegram-link-button"
                    >
                      <Copy className="w-4 h-4 ml-1" />
                      کپی
                    </Button>
                  </div>
                  <div className="p-3 bg-blue-50 rounded-lg border border-blue-200 text-xs break-all">
                    {`https://t.me/${telegramBotUsername}?start=${user.referral_code}`}
                  </div>
                  <p className="text-xs text-gray-500 text-center">
                    💡 لینک تلگرامی را در تلگرام به اشتراک بگذارید
                  </p>
                </div>
              </>
            ) : (
              /* پیام برای کاربران غیرفعال */
              <div className="p-8 bg-gradient-to-r from-yellow-50 to-orange-50 rounded-xl border-2 border-yellow-300 text-center">
                <div className="mb-4">
                  <div className="w-16 h-16 mx-auto bg-yellow-400 rounded-full flex items-center justify-center">
                    <Share2 className="w-8 h-8 text-white" />
                  </div>
                </div>
                <h3 className="text-lg font-bold text-gray-800 mb-2">
                  🔒 لینک‌های دعوت در انتظار فعالسازی
                </h3>
                <p className="text-gray-700 mb-4">
                  بعد از فعال شدن حساب شما، لینک‌های دعوت در اینجا نمایش داده خواهند شد.
                </p>
                <p className="text-sm text-gray-600">
                  💡 پس از فعالسازی، لینک‌های دعوت شما در تلگرام نیز برای شما ارسال می‌شوند.
                </p>
              </div>
            )}
          </CardContent>
        </Card>

        {/* Conditional Content Based on User Status */}
        {profile?.status === "pending" ? (
          <CryptoPaymentPage />
        ) : (
          <Tabs defaultValue="stats" className="space-y-6">
            <TabsList className="flex flex-wrap gap-2 bg-white shadow-md rounded-xl p-3 h-auto">
              <TabsTrigger value="stats" className="flex-1 min-w-[120px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-emerald-50 data-[state=active]:bg-emerald-600 data-[state=active]:text-white" data-testid="stats-tab">
                <TrendingUp className="w-4 h-4 ml-2" />
                داشبورد
              </TabsTrigger>
              <TabsTrigger value="withdrawal" className="flex-1 min-w-[120px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-emerald-50 data-[state=active]:bg-emerald-600 data-[state=active]:text-white" data-testid="withdrawal-tab">
                <Wallet className="w-4 h-4 ml-2" />
                برداشت
              </TabsTrigger>
              <TabsTrigger value="transactions" className="flex-1 min-w-[120px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-emerald-50 data-[state=active]:bg-emerald-600 data-[state=active]:text-white" data-testid="transactions-tab">
                <DollarSign className="w-4 h-4 ml-2" />
                تراکنش‌ها
              </TabsTrigger>
              <TabsTrigger value="faq" className="flex-1 min-w-[120px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-blue-50 data-[state=active]:bg-blue-600 data-[state=active]:text-white">
                <BookOpen className="w-4 h-4 ml-2" />
                توضیحات
              </TabsTrigger>
              <TabsTrigger value="tutorial" className="flex-1 min-w-[120px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-green-50 data-[state=active]:bg-green-600 data-[state=active]:text-white">
                <GraduationCap className="w-4 h-4 ml-2" />
                آموزش
              </TabsTrigger>
              <TabsTrigger value="messages" className="flex-1 min-w-[120px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-purple-50 data-[state=active]:bg-purple-600 data-[state=active]:text-white">
                <MessageSquare className="w-4 h-4 ml-2" />
                پیام‌ها
              </TabsTrigger>
              <TabsTrigger value="support" className="flex-1 min-w-[120px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-cyan-50 data-[state=active]:bg-cyan-600 data-[state=active]:text-white">
                <MessageCircle className="w-4 h-4 ml-2" />
                پشتیبانی
              </TabsTrigger>
              <TabsTrigger value="settings" className="flex-1 min-w-[120px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-gray-50 data-[state=active]:bg-gray-600 data-[state=active]:text-white">
                <Settings className="w-4 h-4 ml-2" />
                تنظیمات
              </TabsTrigger>
            </TabsList>

          <TabsContent value="stats" data-testid="stats-content">
            <Card className="shadow-lg border-0">
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <TrendingUp className="w-5 h-5 text-emerald-600" />
                  تعداد افراد در هر سطح
                </CardTitle>
                <CardDescription>هرم سه‌تایی - هر نفر حداکثر 3 زیرمجموعه مستقیم</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {[
                    { level: 1, max: 1, label: "سطح ۱ (خودتان)" },
                    { level: 2, max: 3, label: "سطح ۲ (دعوت مستقیم)" },
                    { level: 3, max: 9, label: "سطح ۳" },
                    { level: 4, max: 27, label: "سطح ۴" },
                    { level: 5, max: 81, label: "سطح ۵" },
                    { level: 6, max: 243, label: "سطح ۶" },
                    { level: 7, max: 729, label: "سطح ۷" }
                  ].map((item) => (
                    <div key={item.level} className="flex items-center justify-between p-4 bg-gradient-to-r from-emerald-50 to-teal-50 rounded-lg" data-testid={`level-${item.level}-count`}>
                      <div>
                        <span className="font-semibold text-gray-700">{item.label}</span>
                        <span className="text-xs text-gray-500 mr-2">(حداکثر: {item.max})</span>
                      </div>
                      <Badge variant="secondary" className="text-base px-4 py-1">
                        {profile?.level_counts[`level_${item.level}`] || 0} نفر
                      </Badge>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="transactions" data-testid="transactions-content">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              {/* Direct Rewards */}
              <Card className="shadow-lg border-0">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2 text-lg">
                    <Users className="w-5 h-5 text-emerald-600" />
                    پاداش دعوت مستقیم
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  {transactions.filter(tx => tx.reward_type === 'direct').length > 0 ? (
                    <div className="space-y-3">
                      {transactions.filter(tx => tx.reward_type === 'direct').map((tx) => (
                        <div
                          key={tx.transaction_id}
                          className="flex items-center justify-between p-3 bg-gradient-to-r from-blue-50 to-cyan-50 rounded-lg"
                          data-testid="direct-reward-item"
                        >
                          <div>
                            <p className="font-semibold text-gray-800 text-sm">{tx.description}</p>
                            <p className="text-xs text-gray-500">
                              {new Date(tx.timestamp).toLocaleDateString('fa-IR')}
                            </p>
                          </div>
                          <Badge className="bg-blue-600 text-white px-3 py-1">
                            +{tx.amount.toFixed(0)}
                          </Badge>
                        </div>
                      ))}
                    </div>
                  ) : (
                    <p className="text-gray-500 text-center py-4 text-sm">هنوز پاداش مستقیمی دریافت نکرده‌اید</p>
                  )}
                </CardContent>
              </Card>

              {/* Level Rewards */}
              <Card className="shadow-lg border-0">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2 text-lg">
                    <Network className="w-5 h-5 text-teal-600" />
                    پاداش سطوح (۳، ۵، ۷)
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  {transactions.filter(tx => tx.reward_type.startsWith('level_')).length > 0 ? (
                    <div className="space-y-3">
                      {transactions.filter(tx => tx.reward_type.startsWith('level_')).map((tx) => (
                        <div
                          key={tx.transaction_id}
                          className="flex items-center justify-between p-3 bg-gradient-to-r from-teal-50 to-emerald-50 rounded-lg"
                          data-testid="level-reward-item"
                        >
                          <div>
                            <p className="font-semibold text-gray-800 text-sm">{tx.description}</p>
                            <p className="text-xs text-gray-500">
                              {new Date(tx.timestamp).toLocaleDateString('fa-IR')}
                            </p>
                          </div>
                          <Badge className="bg-teal-600 text-white px-3 py-1">
                            +{tx.amount.toFixed(0)}
                          </Badge>
                        </div>
                      ))}
                    </div>
                  ) : (
                    <p className="text-gray-500 text-center py-4 text-sm">هنوز پاداش سطحی دریافت نکرده‌اید</p>
                  )}
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="withdrawal" data-testid="withdrawal-content">
            <WithdrawalPage user={profile} />
          </TabsContent>

          <TabsContent value="messages">
            <MessagingSystem userType="user" />
          </TabsContent>

          <TabsContent value="faq">
            <IframePage 
              configKey="faq_url"
              title="صفحه توضیحات"
              description="برای مشاهده اطلاعات تکمیلی و پاسخ سوالات متداول، به صفحه زیر مراجعه کنید"
              icon={<BookOpen className="w-6 h-6" />}
            />
          </TabsContent>

          <TabsContent value="tutorial">
            <IframePage 
              configKey="tutorial_url"
              title="صفحه آموزش"
              description="برای یادگیری نحوه استفاده از سیستم و آشنایی با امکانات، به صفحه آموزش مراجعه کنید"
              icon={<GraduationCap className="w-6 h-6" />}
            />
          </TabsContent>

          <TabsContent value="support">
            <TelegramSupport />
          </TabsContent>

          <TabsContent value="settings">
            <UserSettings user={profile} />
          </TabsContent>
        </Tabs>
        )}
      </div>
    </div>
  );
}

export default UserDashboard;
