import { useState } from "react";
import axios from "axios";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { toast } from "sonner";
import { Link } from "react-router-dom";
import { Mail, Lock, LogIn } from "lucide-react";
import LoginTabs from "@/components/LoginTabs";

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

function Login({ onLogin }) {
  const [email, setEmail] = useState("");
  const [password, setPassword] = useState("");
  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);

    try {
      console.log("Sending login request to:", `${API}/auth/login`);
      
      const response = await axios.post(`${API}/auth/login`, {
        email,
        password,
      }, {
        headers: {
          'Content-Type': 'application/json',
        }
      });

      console.log("Login response:", response.data);
      console.log("Response type:", typeof response.data);
      
      // چک کردن که HTML نیست
      if (typeof response.data === 'string') {
        console.error("Received HTML instead of JSON!");
        toast.error("خطا: سرور پاسخ نامعتبر داد");
        return;
      }
      
      // چک کردن token
      if (!response.data || !response.data.token) {
        console.error("No token in response!");
        toast.error("خطا: توکن دریافت نشد");
        return;
      }

      toast.success("ورود موفقیت‌آمیز بود");
      
      // ارسال user data به App
      const userData = response.data.user || response.data;
      console.log("Calling onLogin with:", userData);
      onLogin(response.data.token, userData);
      
    } catch (error) {
      console.error("Login error:", error);
      console.error("Error response:", error.response);
      toast.error(error.response?.data?.detail || "خطا در ورود");
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen flex flex-col items-center justify-center bg-gradient-to-br from-emerald-50 via-teal-50 to-cyan-50 p-4">
      <Card className="w-full max-w-md shadow-2xl border-0 backdrop-blur-sm bg-white/90" data-testid="login-card">
        <CardHeader className="text-center space-y-2">
          <div className="mx-auto w-16 h-16 bg-gradient-to-br from-emerald-500 to-teal-600 rounded-2xl flex items-center justify-center mb-4 shadow-lg">
            <LogIn className="w-8 h-8 text-white" />
          </div>
          <CardTitle className="text-3xl font-bold text-emerald-900">ورود به سیستم</CardTitle>
          <CardDescription className="text-base">به سیستم MLM خوش آمدید</CardDescription>
        </CardHeader>
        <CardContent>
          <form onSubmit={handleSubmit} className="space-y-5">
            <div className="space-y-2">
              <Label htmlFor="email" className="text-sm font-semibold text-gray-700">ایمیل</Label>
              <div className="relative">
                <Mail className="absolute right-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                <Input
                  id="email"
                  type="email"
                  placeholder="example@email.com"
                  value={email}
                  onChange={(e) => setEmail(e.target.value)}
                  required
                  className="pr-10 h-12 border-gray-200 focus:border-emerald-500 focus:ring-emerald-500"
                  data-testid="email-input"
                />
              </div>
            </div>
            <div className="space-y-2">
              <Label htmlFor="password" className="text-sm font-semibold text-gray-700">رمز عبور</Label>
              <div className="relative">
                <Lock className="absolute right-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                <Input
                  id="password"
                  type="password"
                  placeholder="••••••••"
                  value={password}
                  onChange={(e) => setPassword(e.target.value)}
                  required
                  className="pr-10 h-12 border-gray-200 focus:border-emerald-500 focus:ring-emerald-500"
                  data-testid="password-input"
                />
              </div>
            </div>
            <Button
              type="submit"
              disabled={loading}
              className="w-full h-12 bg-gradient-to-r from-emerald-600 to-teal-600 hover:from-emerald-700 hover:to-teal-700 text-white font-semibold rounded-xl shadow-lg hover:shadow-xl transition-all text-base"
              data-testid="login-submit-button"
            >
              {loading ? "در حال ورود..." : "ورود"}
            </Button>
          </form>
          <div className="mt-6 text-center">
            <p className="text-gray-600">
              حساب کاربری ندارید؟{" "}
              <Link to="/register" className="text-emerald-600 hover:text-emerald-700 font-semibold" data-testid="register-link">
                ثبت‌نام کنید
              </Link>
            </p>
          </div>
        </CardContent>
      </Card>
      
      {/* تب‌های اطلاعاتی */}
      <LoginTabs />
    </div>
  );
}

export default Login;
