import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { toast } from 'sonner';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Switch } from './ui/switch';
import { Tabs, TabsContent, TabsList, TabsTrigger } from './ui/tabs';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger, DialogDescription } from './ui/dialog';
import { CheckCircle, Clock, RefreshCw, Settings, Key, Send, Zap } from 'lucide-react';

const API = process.env.REACT_APP_BACKEND_URL || '';

export default function AdminCryptoManagement() {
  const [payments, setPayments] = useState([]);
  const [withdrawals, setWithdrawals] = useState([]);
  const [cryptoSettings, setCryptoSettings] = useState(null);
  const [selectedPayment, setSelectedPayment] = useState(null);
  const [transactionHash, setTransactionHash] = useState('');
  const [withdrawalHash, setWithdrawalHash] = useState('');
  const [rejectReason, setRejectReason] = useState('');
  const [trongridApiKey, setTrongridApiKey] = useState('');
  const [tronAddress, setTronAddress] = useState('');
  const [tronEnabled, setTronEnabled] = useState(true);
  const [bscAddress, setBscAddress] = useState('');
  const [bscEnabled, setBscEnabled] = useState(false);
  const [checking, setChecking] = useState(false);
  
  // تنظیمات کارمزد برداشت
  const [minWithdrawal, setMinWithdrawal] = useState(10);
  const [feeType, setFeeType] = useState('fixed');
  const [feeAmount, setFeeAmount] = useState(0);
  
  // Hot Wallet برای auto-transfer
  const [tronPrivateKey, setTronPrivateKey] = useState('');
  const [saveTronKey, setSaveTronKey] = useState(false);
  const [bscPrivateKey, setBscPrivateKey] = useState('');
  const [saveBscKey, setSaveBscKey] = useState(false);
  
  // برای modal auto-transfer (فقط برای وقتی که Private Key ذخیره نشده)
  const [selectedWithdrawal, setSelectedWithdrawal] = useState(null);
  const [modalPrivateKey, setModalPrivateKey] = useState('');
  const [modalSaveKey, setModalSaveKey] = useState(false);
  const [isTransferring, setIsTransferring] = useState(false);
  const [validatingId, setValidatingId] = useState(null);
  
  // موجودی کیف پول‌ها
  const [tronBalance, setTronBalance] = useState(null);
  const [bscBalance, setBscBalance] = useState(null);
  const [checkingBalance, setCheckingBalance] = useState(false);

  useEffect(() => {
    fetchData();
  }, []);

  const getAuthHeaders = () => {
    const token = localStorage.getItem('token');
    return { Authorization: `Bearer ${token}` };
  };

  const fetchData = async () => {
    try {
      const headers = getAuthHeaders();
      const [paymentsRes, withdrawalsRes, settingsRes] = await Promise.all([
        axios.get(`${API}/api/admin/crypto-payments`, { headers }),
        axios.get(`${API}/api/admin/withdrawals`, { headers }),
        axios.get(`${API}/api/admin/settings/crypto`, { headers })
      ]);
      setPayments(paymentsRes.data);
      setWithdrawals(withdrawalsRes.data);
      setCryptoSettings(settingsRes.data);
      
      // تنظیمات TRON
      setTrongridApiKey(settingsRes.data?.trongrid_api_key || '');
      setTronAddress(settingsRes.data?.tron_deposit_address || '');
      setTronEnabled(settingsRes.data?.tron_enabled !== false);
      
      // تنظیمات BSC
      setBscAddress(settingsRes.data?.bsc_deposit_address || '');
      setBscEnabled(settingsRes.data?.bsc_enabled || false);
      
      // تنظیمات کارمزد و حداقل برداشت
      setMinWithdrawal(settingsRes.data?.min_withdrawal_amount || 10);
      setFeeType(settingsRes.data?.withdrawal_fee_type || 'fixed');
      setFeeAmount(settingsRes.data?.withdrawal_fee_amount || 0);
      
      // بارگذاری Private Keys ذخیره شده (اگر موجود باشد)
      if (settingsRes.data?.tron_hot_wallet_private_key) {
        setTronPrivateKey(settingsRes.data.tron_hot_wallet_private_key);
        setSaveTronKey(true);
      }
      if (settingsRes.data?.bsc_hot_wallet_private_key) {
        setBscPrivateKey(settingsRes.data.bsc_hot_wallet_private_key);
        setSaveBscKey(true);
      }
    } catch (error) {
      toast.error('خطا در دریافت اطلاعات');
      console.error(error);
    }
  };

  const checkTransactions = async () => {
    setChecking(true);
    try {
      const headers = getAuthHeaders();
      const res = await axios.post(`${API}/api/admin/crypto-payments/check-transactions`, {}, { headers });
      toast.success(`✅ ${res.data.confirmed_payments} پرداخت تایید شد از ${res.data.checked_transactions} تراکنش`);
      fetchData();
    } catch (error) {
      toast.error('خطا در بررسی تراکنش‌ها');
    } finally {
      setChecking(false);
    }
  };

  const confirmPayment = async (paymentId) => {
    if (!transactionHash.trim()) {
      toast.error('Hash تراکنش را وارد کنید');
      return;
    }

    try {
      const headers = getAuthHeaders();
      await axios.post(
        `${API}/api/admin/crypto-payment/${paymentId}/confirm?transaction_hash=${transactionHash}`,
        {},
        { headers }
      );
      toast.success('✅ پرداخت تایید شد و کاربر فعال شد');
      setSelectedPayment(null);
      setTransactionHash('');
      fetchData();
    } catch (error) {
      toast.error(error.response?.data?.detail || 'خطا در تایید');
    }
  };

  const approveWithdrawal = async (withdrawalId) => {
    if (!withdrawalHash.trim()) {
      toast.error('Hash تراکنش را وارد کنید');
      return;
    }

    try {
      const headers = getAuthHeaders();
      await axios.put(
        `${API}/api/admin/withdrawals/${withdrawalId}/approve`, 
        { transaction_hash: withdrawalHash },
        { headers }
      );
      toast.success('✅ برداشت تایید شد');
      setWithdrawalHash('');
      fetchData();
    } catch (error) {
      toast.error('خطا در تایید');
    }
  };

  const rejectWithdrawal = async (withdrawalId) => {
    try {
      const headers = getAuthHeaders();
      await axios.put(
        `${API}/api/admin/withdrawals/${withdrawalId}/reject?reason=${rejectReason || 'رد شده'}`,
        {},
        { headers }
      );
      toast.success('✅ درخواست رد شد و موجودی بازگشت');
      setRejectReason('');
      fetchData();
    } catch (error) {
      toast.error('خطا در رد درخواست');
    }
  };

  const checkWalletBalance = async (network) => {
    const privateKey = network === 'TRON' ? tronPrivateKey : bscPrivateKey;
    
    if (!privateKey) {
      toast.error(`❌ Private Key ${network} ذخیره نشده!`);
      return;
    }
    
    setCheckingBalance(true);
    try {
      const headers = getAuthHeaders();
      const response = await axios.post(
        `${API}/api/admin/wallet/check-balance`,
        {
          private_key: privateKey,
          network: network
        },
        { headers }
      );
      
      if (network === 'TRON') {
        setTronBalance(response.data);
        toast.success(`✅ موجودی TRON: ${response.data.usdt_balance.toFixed(2)} USDT | ${response.data.trx_balance.toFixed(2)} TRX`);
      } else {
        setBscBalance(response.data);
        toast.success(`✅ موجودی BSC: ${response.data.usdt_balance.toFixed(2)} USDT | ${response.data.bnb_balance.toFixed(4)} BNB`);
      }
    } catch (error) {
      console.error(error);
      toast.error(error.response?.data?.detail || 'خطا در دریافت موجودی');
    } finally {
      setCheckingBalance(false);
    }
  };

  const validateWithdrawal = async (withdrawalId) => {
    setValidatingId(withdrawalId);
    const loadingToast = toast.loading('⏳ در حال بررسی اعتبار...');
    
    try {
      const token = localStorage.getItem('token');
      const response = await axios.post(
        `${API}/api/admin/withdrawals/${withdrawalId}/validate`,
        {},
        {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          }
        }
      );
      
      toast.dismiss(loadingToast);
      
      if (response.data.valid) {
        toast.success('✅ درخواست معتبر است!', { duration: 3000 });
      } else {
        toast.error('❌ امتیاز کاربر کافی نیست!', { duration: 3000 });
      }
      
      // رفرش لیست
      await fetchData();
    } catch (error) {
      toast.dismiss(loadingToast);
      const errorMsg = error.response?.data?.detail || error.message || 'خطا در بررسی اعتبار';
      toast.error(`❌ ${errorMsg}`, { duration: 3000 });
    } finally {
      setValidatingId(null);
    }
  };

  const rejectWithdrawalRequest = async (withdrawalId, reason) => {
    if (!window.confirm(`آیا مطمئن هستید؟\n\nدلیل رد: ${reason || 'رد شده توسط ادمین'}\n\nموجودی به کاربر بازگشت داده می‌شود.`)) {
      return;
    }
    
    const loadingToast = toast.loading('⏳ در حال رد کردن...');
    
    try {
      const token = localStorage.getItem('token');
      const response = await axios.put(
        `${API}/api/admin/withdrawals/${withdrawalId}/reject?reason=${encodeURIComponent(reason || 'رد شده توسط ادمین')}`,
        null,
        {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          }
        }
      );
      
      toast.dismiss(loadingToast);
      toast.success('✅ درخواست رد شد و موجودی به کاربر بازگشت', { duration: 4000 });
      
      // رفرش داده‌ها
      setRejectReason('');
      await fetchData();
    } catch (error) {
      toast.dismiss(loadingToast);
      const errorMsg = error.response?.data?.detail || error.message || 'خطا در رد';
      toast.error(`❌ ${errorMsg}`, { duration: 4000 });
    }
  };

  const deleteWithdrawalRequest = async (withdrawalId) => {
    if (!window.confirm('آیا مطمئن هستید؟\n\nاین درخواست حذف می‌شود و موجودی به کاربر بازمی‌گردد.')) {
      return;
    }
    
    const loadingToast = toast.loading('⏳ در حال حذف...');
    
    try {
      const token = localStorage.getItem('token');
      const response = await axios.delete(
        `${API}/api/admin/withdrawals/${withdrawalId}`,
        {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          }
        }
      );
      
      toast.dismiss(loadingToast);
      toast.success('✅ درخواست حذف شد و موجودی به کاربر بازگشت', { duration: 4000 });
      
      // رفرش داده‌ها
      await fetchData();
    } catch (error) {
      toast.dismiss(loadingToast);
      const errorMsg = error.response?.data?.detail || error.message || 'خطا در حذف';
      toast.error(`❌ ${errorMsg}`, { duration: 4000 });
    }
  };

  // انتقال هوشمند - اگر Private Key ذخیره شده باشد بدون modal، اگر نه با modal
  const smartAutoTransfer = async (withdrawal) => {
    const network = withdrawal.network;
    const savedPrivateKey = (network === 'USDT-TRC20' || network === 'TRON') ? tronPrivateKey : bscPrivateKey;
    
    // اگر Private Key ذخیره شده، مستقیماً انجام بده
    if (savedPrivateKey) {
      const amount = withdrawal.final_transfer_amount || withdrawal.amount;
      if (!window.confirm(`آیا مطمئن هستید؟\n\nانتقال ${amount?.toFixed(2)} USDT\nبه: ${withdrawal.user_email}\nشبکه: ${network}`)) {
        return;
      }
      
      setIsTransferring(true);
      const loadingToast = toast.loading('⏳ در حال ارسال تراکنش...');
      
      try {
        const token = localStorage.getItem('token');
        const response = await axios.post(
          `${API}/api/admin/withdrawals/${withdrawal.withdrawal_id}/auto-transfer`,
          {
            private_key: savedPrivateKey,
            save_key: false
          },
          {
            headers: {
              'Authorization': `Bearer ${token}`,
              'Content-Type': 'application/json'
            }
          }
        );
        
        toast.dismiss(loadingToast);
        toast.success(`✅ تراکنش ارسال شد!\nHash: ${response.data.tx_hash.substring(0, 10)}...`, { duration: 5000 });
        
        // رفرش داده‌ها
        await fetchData();
      } catch (error) {
        toast.dismiss(loadingToast);
        const errorMsg = error.response?.data?.detail || error.message || 'خطا در ارسال تراکنش';
        toast.error(`❌ ${errorMsg}`, { duration: 5000 });
      } finally {
        setIsTransferring(false);
      }
    } else {
      // اگر Private Key ذخیره نشده، Modal نشان بده
      setSelectedWithdrawal(withdrawal);
      setModalPrivateKey('');
      setModalSaveKey(false);
    }
  };

  // ارسال تراکنش از Modal (وقتی Private Key ذخیره نشده)
  const executeModalTransfer = async () => {
    if (!modalPrivateKey.trim()) {
      toast.error('❌ Private Key را وارد کنید');
      return;
    }
    
    setIsTransferring(true);
    const loadingToast = toast.loading('⏳ در حال ارسال تراکنش...');
    
    try {
      const token = localStorage.getItem('token');
      const response = await axios.post(
        `${API}/api/admin/withdrawals/${selectedWithdrawal.withdrawal_id}/auto-transfer`,
        {
          private_key: modalPrivateKey,
          save_key: modalSaveKey
        },
        {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          }
        }
      );
      
      toast.dismiss(loadingToast);
      toast.success(`✅ تراکنش ارسال شد!\nHash: ${response.data.tx_hash.substring(0, 10)}...`, { duration: 5000 });
      
      // اگر کلید ذخیره شد، state را آپدیت کن
      if (modalSaveKey) {
        const network = selectedWithdrawal.network;
        if (network === 'USDT-TRC20' || network === 'TRON') {
          setTronPrivateKey(modalPrivateKey);
          setSaveTronKey(true);
        } else {
          setBscPrivateKey(modalPrivateKey);
          setSaveBscKey(true);
        }
      }
      
      setSelectedWithdrawal(null);
      setModalPrivateKey('');
      await fetchData();
    } catch (error) {
      toast.dismiss(loadingToast);
      const errorMsg = error.response?.data?.detail || error.message || 'خطا در ارسال تراکنش';
      toast.error(`❌ ${errorMsg}`, { duration: 5000 });
    } finally {
      setIsTransferring(false);
    }
  };

  // انتقال همه برداشت‌های معتبر
  const transferAllValidWithdrawals = async () => {
    const validWithdrawals = withdrawals.filter(w => 
      w.status === 'pending_admin' && 
      w.validation_status === 'valid'
    );
    
    if (validWithdrawals.length === 0) {
      toast.error('❌ هیچ برداشت معتبری برای انتقال وجود ندارد');
      return;
    }
    
    // گروه‌بندی بر اساس شبکه
    const tronWithdrawals = validWithdrawals.filter(w => w.network === 'USDT-TRC20' || w.network === 'TRON');
    const bscWithdrawals = validWithdrawals.filter(w => w.network === 'USDT-BEP20' || w.network === 'BSC');
    
    // چک Private Keys
    if (tronWithdrawals.length > 0 && !tronPrivateKey) {
      toast.error('❌ Private Key TRON ذخیره نشده! لطفاً ابتدا در تنظیمات ذخیره کنید.');
      return;
    }
    if (bscWithdrawals.length > 0 && !bscPrivateKey) {
      toast.error('❌ Private Key BSC ذخیره نشده! لطفاً ابتدا در تنظیمات ذخیره کنید.');
      return;
    }
    
    // تایید
    const totalAmount = validWithdrawals.reduce((sum, w) => sum + (w.final_transfer_amount || 0), 0);
    if (!window.confirm(
      `آیا مطمئن هستید؟\n\n` +
      `تعداد: ${validWithdrawals.length} برداشت\n` +
      `مجموع: ${totalAmount.toFixed(2)} USDT\n` +
      `TRON: ${tronWithdrawals.length}\n` +
      `BSC: ${bscWithdrawals.length}`
    )) {
      return;
    }
    
    setIsTransferring(true);
    
    // انتقال یکی یکی
    let success = 0;
    let failed = 0;
    const results = [];
    
    for (const w of validWithdrawals) {
      try {
        const pk = (w.network === 'USDT-TRC20' || w.network === 'TRON') ? tronPrivateKey : bscPrivateKey;
        const headers = getAuthHeaders();
        
        const response = await axios.post(
          `${API}/api/admin/withdrawals/${w.withdrawal_id}/auto-transfer`,
          {
            private_key: pk,
            save_key: false
          },
          { headers }
        );
        
        success++;
        results.push({ email: w.user_email, success: true, tx: response.data.tx_hash });
        
        // تاخیر 2 ثانیه بین تراکنش‌ها
        await new Promise(resolve => setTimeout(resolve, 2000));
      } catch (error) {
        failed++;
        results.push({ email: w.user_email, success: false, error: error.response?.data?.detail });
        console.error(error);
      }
    }
    
    setIsTransferring(false);
    
    // نمایش نتیجه
    if (failed === 0) {
      toast.success(`✅ همه تراکنش‌ها موفق بود! (${success} عدد)`);
    } else {
      toast.error(`⚠️ موفق: ${success} | ناموفق: ${failed}`);
    }
    
    await fetchData();
  };

  const updateCryptoSettings = async (e) => {
    e.preventDefault();
    
    // بررسی حداقل یک شبکه فعال
    if (!tronEnabled && !bscEnabled) {
      toast.error('❌ حداقل یک شبکه باید فعال باشد');
      return;
    }
    
    // Validation برای TRON (اگر فعال است)
    if (tronEnabled) {
      if (!tronAddress) {
        toast.error('❌ لطفاً آدرس TRON را وارد کنید');
        return;
      }
      if (!tronAddress.startsWith('T')) {
        toast.error('❌ آدرس TRON باید با حرف T شروع شود');
        return;
      }
      if (tronAddress.length !== 34) {
        toast.error(`❌ آدرس TRON باید دقیقاً 34 کاراکتر باشد (فعلی: ${tronAddress.length})`);
        return;
      }
    }
    
    // Validation برای BSC (اگر فعال است)
    if (bscEnabled) {
      if (!bscAddress) {
        toast.error('❌ لطفاً آدرس BSC را وارد کنید');
        return;
      }
      if (!bscAddress.startsWith('0x')) {
        toast.error('❌ آدرس BSC باید با 0x شروع شود');
        return;
      }
      if (bscAddress.length !== 42) {
        toast.error(`❌ آدرس BSC باید دقیقاً 42 کاراکتر باشد (فعلی: ${bscAddress.length})`);
        return;
      }
    }
    
    try {
      const headers = getAuthHeaders();
      const response = await axios.put(`${API}/api/admin/settings/crypto`, {
        // TRON
        trongrid_api_key: trongridApiKey || null,
        tron_deposit_address: tronAddress || null,
        tron_enabled: tronEnabled,
        // BSC (بدون API key - از Public RPC استفاده می‌کند)
        bsc_deposit_address: bscAddress || null,
        bsc_enabled: bscEnabled,
        // کارمزد برداشت و حداقل مبلغ
        min_withdrawal_amount: parseFloat(minWithdrawal) || 10,
        withdrawal_fee_type: feeType,
        withdrawal_fee_amount: parseFloat(feeAmount) || 0,
        // Hot Wallet (فقط اگر تیک ذخیره خورده باشد)
        tron_hot_wallet_private_key: saveTronKey && tronPrivateKey ? tronPrivateKey : null,
        save_tron_private_key: saveTronKey,
        bsc_hot_wallet_private_key: saveBscKey && bscPrivateKey ? bscPrivateKey : null,
        save_bsc_private_key: saveBscKey,
        // Legacy
        auto_check_enabled: cryptoSettings?.auto_check_enabled || false,
        min_confirmation: cryptoSettings?.min_confirmation || 19
      }, { headers });
      
      toast.success('✅ تنظیمات با موفقیت ذخیره شد!');
      
      // دوباره تنظیمات را دریافت کنیم تا مطمئن شویم ذخیره شده
      await fetchData();
      
      console.log('✅ Settings updated:', response.data);
    } catch (error) {
      console.error('❌ Settings update error:', error);
      toast.error(error.response?.data?.detail || 'خطا در به‌روزرسانی');
    }
  };

  // createHotWallet function removed - functionality moved to settings

  return (
    <div className="p-6 space-y-6">
      <Tabs defaultValue="payments" className="w-full">
        <TabsList className="grid w-full grid-cols-3">
          <TabsTrigger value="payments">پرداخت‌ها</TabsTrigger>
          <TabsTrigger value="withdrawals">برداشت‌ها</TabsTrigger>
          <TabsTrigger value="settings">تنظیمات</TabsTrigger>
        </TabsList>

        {/* پرداخت‌ها */}
        <TabsContent value="payments">
          <Card>
            <CardHeader>
              <div className="flex justify-between items-center">
                <CardTitle>
                  مدیریت پرداخت‌ها ({payments.filter(p => p.status === 'pending').length} در انتظار)
                </CardTitle>
                <Button 
                  onClick={checkTransactions} 
                  disabled={checking}
                  variant="outline"
                  className="border-green-500 text-green-600 hover:bg-green-50"
                >
                  {checking ? (
                    <>
                      <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-green-500 mr-2"></div>
                      در حال بررسی...
                    </>
                  ) : (
                    <>
                      <RefreshCw className="w-4 h-4 mr-2" />
                      بررسی خودکار تراکنش‌ها
                    </>
                  )}
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {payments.length === 0 ? (
                  <div className="text-center py-8 text-gray-500">
                    هیچ پرداختی ثبت نشده است
                  </div>
                ) : (
                  payments.map((payment) => (
                    <div key={payment.payment_id} className="border rounded p-4">
                      <div className="flex justify-between items-start">
                        <div>
                          <div className="font-semibold">{payment.user_email}</div>
                          <div className="text-sm text-gray-600">{payment.amount} USDT - {payment.network}</div>
                          <div className="text-xs text-gray-500 mt-1">
                            {new Date(payment.created_at).toLocaleString('fa-IR')}
                          </div>
                        </div>
                        <span className={`px-3 py-1 rounded-full text-sm ${
                          payment.status === 'confirmed' ? 'bg-green-100 text-green-700' : 'bg-amber-100 text-amber-700'
                        }`}>
                          {payment.status === 'confirmed' ? 'تایید شده' : 'در انتظار'}
                        </span>
                      </div>
                      {payment.status === 'pending' && (
                        <div className="mt-3 flex gap-2">
                          <Input
                            placeholder="Hash تراکنش"
                            value={selectedPayment === payment.payment_id ? transactionHash : ''}
                            onChange={(e) => {
                              setSelectedPayment(payment.payment_id);
                              setTransactionHash(e.target.value);
                            }}
                          />
                          <Button onClick={() => confirmPayment(payment.payment_id)} size="sm">
                            تایید
                          </Button>
                        </div>
                      )}
                    </div>
                  ))
                )}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        {/* برداشت‌ها */}
        <TabsContent value="withdrawals">
          <Card>
            <CardHeader>
              <CardTitle>🎫 درخواست‌های برداشت ({withdrawals.filter(w => w.status === 'pending_admin').length} در انتظار)</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {withdrawals.length === 0 ? (
                  <div className="text-center py-12 text-gray-500">
                    <p className="text-lg">هیچ درخواستی وجود ندارد</p>
                  </div>
                ) : (
                  withdrawals.map((withdrawal) => (
                  <div key={withdrawal.withdrawal_id} className="border-2 rounded-xl p-6 bg-gradient-to-br from-white to-gray-50 shadow-sm hover:shadow-md transition">
                    <div className="flex justify-between items-start mb-4">
                      <div className="flex-1">
                        <div className="flex items-center gap-2 mb-2">
                          <span className="text-sm font-semibold text-gray-600">👤 {withdrawal.user_email}</span>
                        </div>
                        <div className="text-3xl font-bold text-blue-600 mb-2">
                          {withdrawal.amount} <span className="text-xl">USDT</span>
                        </div>
                        <div className="text-sm text-gray-500 mb-3">
                          🌐 شبکه: <span className="font-semibold">{withdrawal.network === 'USDT-BEP20' ? 'BSC' : 'TRON'}</span>
                        </div>
                        <div className="bg-gray-100 p-3 rounded-lg">
                          <div className="text-xs text-gray-500 mb-1">آدرس کیف پول:</div>
                          <div className="font-mono text-xs break-all text-gray-800">
                            {withdrawal.crypto_address}
                          </div>
                        </div>
                      </div>
                      
                      <span className={`px-4 py-2 rounded-full text-sm font-bold whitespace-nowrap ${
                        withdrawal.status === 'completed' ? 'bg-green-100 text-green-700' :
                        withdrawal.status === 'rejected' ? 'bg-red-100 text-red-700' :
                        'bg-amber-100 text-amber-700'
                      }`}>
                        {withdrawal.status === 'completed' ? '✓ پرداخت شد' : 
                         withdrawal.status === 'rejected' ? '✗ رد شد' : 
                         '⏳ در انتظار'}
                      </span>
                    </div>
                    
                    {/* دکمه‌های اصلی - فقط برای pending */}
                    {withdrawal.status === 'pending_admin' && (
                      <div className="grid grid-cols-2 gap-3 mt-4">
                        <Button 
                          onClick={() => smartAutoTransfer(withdrawal)}
                          disabled={isTransferring}
                          className="bg-green-600 hover:bg-green-700 text-white font-bold py-6 text-lg"
                        >
                          {isTransferring ? (
                            <>
                              <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white ml-2"></div>
                              در حال پرداخت...
                            </>
                          ) : (
                            <>
                              ✅ پرداخت
                            </>
                          )}
                        </Button>
                        
                        <Button 
                          onClick={() => rejectWithdrawalRequest(withdrawal.withdrawal_id, 'رد شده')}
                          disabled={isTransferring}
                          className="bg-red-600 hover:bg-red-700 text-white font-bold py-6 text-lg"
                        >
                          ❌ رد
                        </Button>
                      </div>
                    )}
                    
                    {/* نمایش Hash برای completed */}
                    {withdrawal.status === 'completed' && withdrawal.transaction_hash && (
                      <div className="mt-4 p-3 bg-green-50 border-2 border-green-300 rounded-lg">
                        <div className="text-xs text-green-700 mb-1 font-semibold">✅ تراکنش موفق:</div>
                        <div className="font-mono text-xs break-all text-green-800">{withdrawal.transaction_hash}</div>
                      </div>
                    )}
                    
                    {/* نمایش دلیل رد */}
                    {withdrawal.status === 'rejected' && (
                      <div className="mt-4 p-3 bg-red-50 border-2 border-red-300 rounded-lg">
                        <div className="text-sm text-red-700">❌ این درخواست رد شد</div>
                      </div>
                    )}
                  </div>
                ))
                )}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        {/* Wallet tab removed - functionality moved to settings */}

        {/* تنظیمات */}
        <TabsContent value="settings">
          <div className="space-y-6">
            <form onSubmit={updateCryptoSettings} className="space-y-6">
              {/* TRON Network Settings */}
              <Card className={`border-2 ${tronEnabled ? 'border-blue-300 bg-blue-50/30' : 'border-gray-200 opacity-60'}`}>
                <CardHeader>
                  <div className="flex items-center justify-between">
                    <CardTitle className="flex items-center gap-2 text-lg">
                      <div className="w-10 h-10 rounded-full bg-blue-500 flex items-center justify-center text-white font-bold">
                        T
                      </div>
                      شبکه TRON (TRC20)
                    </CardTitle>
                    <div className="flex items-center gap-2">
                      <Label htmlFor="tron-switch" className="text-sm cursor-pointer">
                        {tronEnabled ? 'فعال' : 'غیرفعال'}
                      </Label>
                      <Switch
                        id="tron-switch"
                        checked={tronEnabled}
                        onCheckedChange={setTronEnabled}
                      />
                    </div>
                  </div>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="bg-blue-50 border border-blue-200 p-4 rounded-lg">
                    <h4 className="font-semibold text-blue-900 mb-2 flex items-center gap-2">
                      <Key className="w-4 h-4" />
                      TronGrid API Key (اختیاری)
                    </h4>
                    <p className="text-xs text-blue-800 mb-3">
                      برای دریافت API Key رایگان به{' '}
                      <a 
                        href="https://www.trongrid.io" 
                        target="_blank" 
                        rel="noopener noreferrer"
                        className="underline font-semibold"
                      >
                        TronGrid.io
                      </a>
                      {' '}مراجعه کنید
                    </p>
                    <Input
                      value={trongridApiKey}
                      onChange={(e) => setTrongridApiKey(e.target.value)}
                      placeholder="API Key برای بررسی خودکار تراکنش‌ها"
                      type="password"
                      disabled={!tronEnabled}
                    />
                  </div>

                  <div>
                    <Label className="text-sm font-semibold">
                      آدرس کیف پول TRON {tronEnabled && <span className="text-red-500">*</span>}
                    </Label>
                    <Input
                      value={tronAddress}
                      onChange={(e) => setTronAddress(e.target.value.trim())}
                      placeholder="مثال: TLyqzVGLV1srkB7dToTAEqgDSfPtXRJZYH"
                      className={`mt-2 font-mono text-sm ${
                        tronAddress && tronAddress.startsWith('T') && tronAddress.length === 34 
                          ? 'border-green-500' 
                          : tronAddress 
                            ? 'border-red-500' 
                            : ''
                      }`}
                      disabled={!tronEnabled}
                      required={tronEnabled}
                    />
                    {tronAddress && (
                      <div className="mt-2 p-2 text-xs space-y-1">
                        <div className={tronAddress.startsWith('T') ? 'text-green-600' : 'text-red-600'}>
                          {tronAddress.startsWith('T') ? '✅' : '❌'} شروع با T
                        </div>
                        <div className={tronAddress.length === 34 ? 'text-green-600' : 'text-red-600'}>
                          {tronAddress.length === 34 ? '✅' : '❌'} 34 کاراکتر (فعلی: {tronAddress.length})
                        </div>
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>

              {/* BSC Network Settings */}
              <Card className={`border-2 ${bscEnabled ? 'border-yellow-300 bg-yellow-50/30' : 'border-gray-200 opacity-60'}`}>
                <CardHeader>
                  <div className="flex items-center justify-between">
                    <CardTitle className="flex items-center gap-2 text-lg">
                      <div className="w-10 h-10 rounded-full bg-yellow-500 flex items-center justify-center text-white font-bold">
                        B
                      </div>
                      شبکه Binance Smart Chain (BEP20)
                    </CardTitle>
                    <div className="flex items-center gap-2">
                      <Label htmlFor="bsc-switch" className="text-sm cursor-pointer">
                        {bscEnabled ? 'فعال' : 'غیرفعال'}
                      </Label>
                      <Switch
                        id="bsc-switch"
                        checked={bscEnabled}
                        onCheckedChange={setBscEnabled}
                      />
                    </div>
                  </div>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="bg-green-50 border border-green-200 p-3 rounded-lg">
                    <p className="text-xs text-green-800">
                      ✅ <strong>بدون نیاز به API Key!</strong> این شبکه از Public RPC استفاده می‌کند و کاملاً رایگان است.
                    </p>
                  </div>

                  <div>
                    <Label className="text-sm font-semibold">
                      آدرس کیف پول BSC {bscEnabled && <span className="text-red-500">*</span>}
                    </Label>
                    <Input
                      value={bscAddress}
                      onChange={(e) => setBscAddress(e.target.value.trim())}
                      placeholder="مثال: 0x742d35Cc6634C0532925a3b844Bc9e7595f0bEb"
                      className={`mt-2 font-mono text-sm ${
                        bscAddress && bscAddress.startsWith('0x') && bscAddress.length === 42 
                          ? 'border-green-500' 
                          : bscAddress 
                            ? 'border-red-500' 
                            : ''
                      }`}
                      disabled={!bscEnabled}
                      required={bscEnabled}
                    />
                    {bscAddress && (
                      <div className="mt-2 p-2 text-xs space-y-1">
                        <div className={bscAddress.startsWith('0x') ? 'text-green-600' : 'text-red-600'}>
                          {bscAddress.startsWith('0x') ? '✅' : '❌'} شروع با 0x
                        </div>
                        <div className={bscAddress.length === 42 ? 'text-green-600' : 'text-red-600'}>
                          {bscAddress.length === 42 ? '✅' : '❌'} 42 کاراکتر (فعلی: {bscAddress.length})
                        </div>
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>

              {/* تنظیمات برداشت */}
              <Card className="border-2 border-purple-300 bg-purple-50/30">
                <CardHeader>
                  <CardTitle className="text-lg text-purple-700">⚙️ تنظیمات برداشت</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label className="text-sm font-semibold">حداقل مبلغ برداشت (USDT)</Label>
                    <Input
                      type="number"
                      step="0.01"
                      min="1"
                      value={minWithdrawal}
                      onChange={(e) => setMinWithdrawal(e.target.value)}
                      placeholder="مثلاً 10"
                      className="mt-2"
                    />
                    <p className="text-xs text-gray-600 mt-2">
                      💡 کاربران نمی‌توانند کمتر از این مبلغ برداشت کنند
                    </p>
                  </div>
                  
                  <div className="border-t pt-4">
                    <Label className="text-sm font-semibold">نوع کارمزد</Label>
                    <select
                      value={feeType}
                      onChange={(e) => setFeeType(e.target.value)}
                      className="w-full border-2 border-gray-300 rounded-lg p-3 mt-2"
                    >
                      <option value="fixed">مبلغ ثابت (USDT)</option>
                      <option value="percentage">درصدی (%)</option>
                    </select>
                  </div>
                  
                  <div>
                    <Label className="text-sm font-semibold">
                      {feeType === 'fixed' ? 'مبلغ کارمزد (USDT)' : 'درصد کارمزد (%)'}
                    </Label>
                    <Input
                      type="number"
                      step="0.01"
                      value={feeAmount}
                      onChange={(e) => setFeeAmount(e.target.value)}
                      placeholder={feeType === 'fixed' ? 'مثلاً 1.5' : 'مثلاً 2'}
                      className="mt-2"
                    />
                    {feeAmount > 0 && (
                      <p className="text-xs text-gray-600 mt-2">
                        مثال: برای برداشت 100 USDT، کارمزد {' '}
                        {feeType === 'fixed' 
                          ? `${feeAmount} USDT` 
                          : `${(100 * feeAmount / 100).toFixed(2)} USDT`
                        } کسر می‌شود
                      </p>
                    )}
                  </div>
                </CardContent>
              </Card>

              {/* Hot Wallet Management */}
              <Card className="border-2 border-orange-300 bg-orange-50/30">
                <CardHeader>
                  <CardTitle className="text-lg text-orange-700">🔐 مدیریت کیف پول گرم (Hot Wallet)</CardTitle>
                  <p className="text-sm text-gray-600 mt-2">
                    برای انتقال خودکار، Private Key را ذخیره کنید تا دیگر نیازی به وارد کردن مجدد نباشد
                  </p>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="bg-yellow-50 border-2 border-yellow-300 p-3 rounded-lg">
                    <p className="text-xs text-yellow-800 font-semibold">
                      ⚠️ توجه: Private Key ها با رمزگذاری ذخیره می‌شوند اما برای امنیت بیشتر، توصیه می‌شود از کیف پول جداگانه برای برداشت‌ها استفاده کنید.
                    </p>
                  </div>

                  <div className="space-y-3">
                    <h4 className="font-semibold text-blue-700">TRON Hot Wallet</h4>
                    <div>
                      <Label className="text-sm">Private Key {saveTronKey && '✅ (ذخیره شده)'}</Label>
                      <Input
                        type="password"
                        value={tronPrivateKey}
                        onChange={(e) => setTronPrivateKey(e.target.value)}
                        placeholder="Private Key کیف پول TRON (بدون 0x)"
                        className="mt-2 font-mono text-xs"
                      />
                    </div>
                    <div className="flex items-center gap-2">
                      <input
                        type="checkbox"
                        id="save-tron-key"
                        checked={saveTronKey}
                        onChange={(e) => setSaveTronKey(e.target.checked)}
                      />
                      <Label htmlFor="save-tron-key" className="text-sm cursor-pointer">
                        ذخیره Private Key (رمزگذاری شده) برای استفاده‌های بعدی
                      </Label>
                    </div>
                    
                    {/* دکمه چک موجودی */}
                    {tronPrivateKey && (
                      <Button
                        type="button"
                        onClick={() => checkWalletBalance('TRON')}
                        disabled={checkingBalance}
                        className="w-full bg-blue-500 hover:bg-blue-600"
                        size="sm"
                      >
                        {checkingBalance ? '🔄 در حال بررسی...' : '💰 بررسی موجودی'}
                      </Button>
                    )}
                    
                    {/* نمایش موجودی */}
                    {tronBalance && (
                      <div className="bg-blue-50 border border-blue-300 p-3 rounded">
                        <p className="text-xs font-bold text-blue-900">📊 موجودی کیف پول:</p>
                        <p className="text-sm text-blue-800 mt-1">
                          💵 USDT: {tronBalance.usdt_balance.toFixed(2)} | 
                          ⚡ TRX: {tronBalance.trx_balance.toFixed(2)}
                        </p>
                        <p className="text-xs text-blue-600 font-mono mt-1">{tronBalance.address}</p>
                      </div>
                    )}
                  </div>

                  <div className="border-t pt-4 space-y-3">
                    <h4 className="font-semibold text-yellow-700">BSC Hot Wallet</h4>
                    <div>
                      <Label className="text-sm">Private Key {saveBscKey && '✅ (ذخیره شده)'}</Label>
                      <Input
                        type="password"
                        value={bscPrivateKey}
                        onChange={(e) => setBscPrivateKey(e.target.value)}
                        placeholder="Private Key کیف پول BSC (با یا بدون 0x)"
                        className="mt-2 font-mono text-xs"
                      />
                    </div>
                    <div className="flex items-center gap-2">
                      <input
                        type="checkbox"
                        id="save-bsc-key"
                        checked={saveBscKey}
                        onChange={(e) => setSaveBscKey(e.target.checked)}
                      />
                      <Label htmlFor="save-bsc-key" className="text-sm cursor-pointer">
                        ذخیره Private Key (رمزگذاری شده) برای استفاده‌های بعدی
                      </Label>
                    </div>
                    
                    {/* دکمه چک موجودی */}
                    {bscPrivateKey && (
                      <Button
                        type="button"
                        onClick={() => checkWalletBalance('BSC')}
                        disabled={checkingBalance}
                        className="w-full bg-yellow-500 hover:bg-yellow-600"
                        size="sm"
                      >
                        {checkingBalance ? '🔄 در حال بررسی...' : '💰 بررسی موجودی'}
                      </Button>
                    )}
                    
                    {/* نمایش موجودی */}
                    {bscBalance && (
                      <div className="bg-yellow-50 border border-yellow-300 p-3 rounded">
                        <p className="text-xs font-bold text-yellow-900">📊 موجودی کیف پول:</p>
                        <p className="text-sm text-yellow-800 mt-1">
                          💵 USDT: {bscBalance.usdt_balance.toFixed(2)} | 
                          ⚡ BNB: {bscBalance.bnb_balance.toFixed(4)}
                        </p>
                        <p className="text-xs text-yellow-600 font-mono mt-1">{bscBalance.address}</p>
                      </div>
                    )}
                  </div>

                  <div className="bg-blue-50 border border-blue-200 p-3 rounded-lg mt-4">
                    <p className="text-xs text-blue-800">
                      💡 <strong>نکته:</strong> اگر Private Key را ذخیره کنید، در بخش برداشت‌ها با یک کلیک می‌توانید تراکنش را ارسال کنید.
                    </p>
                  </div>
                </CardContent>
              </Card>

              <Button type="submit" className="w-full bg-emerald-600 hover:bg-emerald-700 h-12 text-base font-bold">
                💾 ذخیره تمام تنظیمات
              </Button>
            </form>

            {/* راهنما */}
            <Card className="border-emerald-200">
              <CardHeader>
                <CardTitle className="text-emerald-700 text-lg">📚 راهنمای تنظیمات</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4 text-sm">
                <div className="space-y-2">
                  <p className="font-bold text-emerald-700">شبکه TRON (TRC20):</p>
                  <ul className="list-disc list-inside space-y-1 text-gray-700 mr-4">
                    <li>آدرس باید با <code className="bg-gray-100 px-1 rounded">T</code> شروع شود</li>
                    <li>دقیقاً 34 کاراکتر داشته باشد</li>
                    <li>مثال: <code className="bg-gray-100 px-1 rounded text-xs">TLyqzVGLV1srkB7dToTAEqgDSfPtXRJZYH</code></li>
                    <li>کارمزد کم (حدود 1 USDT)</li>
                  </ul>
                </div>
                
                <div className="space-y-2">
                  <p className="font-bold text-yellow-700">شبکه BSC (BEP20):</p>
                  <ul className="list-disc list-inside space-y-1 text-gray-700 mr-4">
                    <li>آدرس باید با <code className="bg-gray-100 px-1 rounded">0x</code> شروع شود</li>
                    <li>دقیقاً 42 کاراکتر داشته باشد</li>
                    <li>مثال: <code className="bg-gray-100 px-1 rounded text-xs">0x742d35Cc6634C0532925a3b844Bc9e7595f0bEb</code></li>
                    <li>کارمزد پایین (حدود 0.5 USDT)</li>
                    <li className="text-green-600 font-semibold">✅ بدون نیاز به API Key - از Public RPC استفاده می‌کند</li>
                  </ul>
                </div>

                <div className="mt-4 p-3 bg-blue-50 border border-blue-200 rounded">
                  <p className="text-xs text-blue-800">
                    <strong>💡 توصیه:</strong> حداقل یک شبکه باید فعال باشد. می‌توانید هر دو شبکه را فعال کنید تا کاربران انتخاب داشته باشند.
                  </p>
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>
      </Tabs>
      
      {/* Modal برای وقتی که Private Key ذخیره نشده */}
      {selectedWithdrawal && (
        <Dialog open={!!selectedWithdrawal} onOpenChange={(open) => !open && setSelectedWithdrawal(null)}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>🔐 انتقال خودکار به {selectedWithdrawal.user_email}</DialogTitle>
              <DialogDescription>
                مبلغ نهایی: {selectedWithdrawal.final_transfer_amount?.toFixed(2)} USDT (پس از کسر کارمزد)
              </DialogDescription>
            </DialogHeader>
            
            <div className="space-y-4 pt-4">
              <div>
                <Label>Private Key ({selectedWithdrawal.network === 'USDT-TRC20' || selectedWithdrawal.network === 'TRON' ? 'TRON' : 'BSC'})</Label>
                <Input
                  type="password"
                  placeholder="Private Key کیف پول خود را وارد کنید"
                  value={modalPrivateKey}
                  onChange={(e) => setModalPrivateKey(e.target.value)}
                  className="font-mono text-xs"
                />
              </div>
              
              <div className="flex items-center gap-2">
                <input
                  type="checkbox"
                  id="modal-save-key"
                  checked={modalSaveKey}
                  onChange={(e) => setModalSaveKey(e.target.checked)}
                />
                <Label htmlFor="modal-save-key" className="text-sm cursor-pointer">
                  ذخیره Private Key (رمزگذاری شده) برای استفاده‌های بعدی
                </Label>
              </div>
              
              <div className="bg-orange-50 border border-orange-300 p-3 rounded">
                <p className="text-xs text-orange-800">
                  ⚠️ <strong>هشدار:</strong> Private Key شما با رمزگذاری ذخیره می‌شود اما همیشه احتیاط کنید.
                </p>
              </div>
              
              <Button 
                onClick={executeModalTransfer}
                className="w-full bg-emerald-600 hover:bg-emerald-700"
                disabled={!modalPrivateKey.trim() || isTransferring}
              >
                {isTransferring ? 'در حال ارسال...' : '✓ تایید و ارسال تراکنش'}
              </Button>
            </div>
          </DialogContent>
        </Dialog>
      )}
    </div>
  );
}
