# 📊 تحلیل جامع سیستم MLM (Multi-Level Marketing)

## خلاصه اجرایی
این سیستم یک پلتفرم بازاریابی شبکه‌ای (MLM) با ساختار هرمی 7 سطحی است که از FastAPI + MongoDB + React ساخته شده و شامل سیستم پاداش چندسطحی، پرداخت کریپتو، و مدیریت برداشت می‌باشد.

---

## 1️⃣ ساختار درخت کاربران (User Tree Structure)

### 🌳 مدل درختی
سیستم از یک **درخت سه‌شاخه‌ای (Ternary Tree)** استفاده می‌کند که در آن:

- **هر کاربر می‌تواند حداکثر 3 فرزند مستقیم داشته باشد**
- **درخت 7 سطحی است** (Level 1 تا Level 7)
- **کاربر ریشه: `seed`** (کاربر ادمین با `user_id="seed"`)
- **جایگذاری بر اساس الگوریتم BFS** (Breadth-First Search)

### 📐 فیلدهای مدل کاربر (User Model)

```javascript
User {
  user_id: string              // شناسه یکتا کاربر
  email: string                // ایمیل
  referrer_id: string          // شناسه معرف (sponsoring line)
  placement_parent_id: string  // شناسه والد در درخت (placement line)
  direct_children: [string]    // آرایه شناسه‌های فرزندان مستقیم (حداکثر 3)
  depth: int                   // عمق در درخت (0 برای seed، 1 برای سطح اول، ...)
  status: string               // "pending" یا "active"
  total_points: float          // مجموع امتیازات/پاداش‌ها
  blocked_direct_reward: bool  // آیا پاداش مستقیم مسدود شده؟
  blocked_level_reward: bool   // آیا پاداش سطحی مسدود شده؟
  is_tree_complete: bool       // آیا هرم 7 سطحی کامل شده؟
}
```

### 🔑 تفاوت Referrer و Placement Parent

این سیستم از **دو خط مجزا** استفاده می‌کند:

1. **Sponsoring Line (Referrer)**: کسی که لینک معرفی را داده → دریافت پاداش مستقیم
2. **Placement Line (Placement Parent)**: کسی که در درخت بالای شماست → دریافت پاداش سطحی

**مثال:**
```
کاربر A کد معرفی خود را به کاربر B می‌دهد
→ referrer_id کاربر B = A (کاربر A پاداش مستقیم دریافت می‌کند)

اما اگر کاربر A قبلاً 3 فرزند داشته باشد:
→ placement_parent_id کاربر B = یک کاربر دیگر در درخت (طبق BFS)
```

### 📊 محدودیت‌های هرم

- **تعداد نودهای هر سطح:**
  - سطح 1: 3 نفر
  - سطح 2: 9 نفر (3×3)
  - سطح 3: 27 نفر (9×3)
  - سطح 4: 81 نفر (27×3)
  - سطح 5: 243 نفر (81×3)
  - سطح 6: 729 نفر (243×3)
  - سطح 7: 2187 نفر (729×3)

- **جمع کل زیرمجموعه‌های یک هرم کامل 7 سطحی:**
  ```
  3 + 9 + 27 + 81 + 243 + 729 + 2187 = 3279 نفر
  ```
  
  **⚠️ توجه:** کد از عدد **1092** استفاده می‌کند که فقط 6 سطح اول را در نظر می‌گیرد:
  ```
  3 + 9 + 27 + 81 + 243 + 729 = 1092
  ```

---

## 2️⃣ محاسبه سطوح (Level Calculation)

### 📏 محاسبه Depth

**Depth** = عمق کاربر در درخت (از 0 شروع می‌شود)

```javascript
// هنگام فعال‌سازی کاربر:
parent = find_user(placement_parent_id)
parent_depth = parent.depth
new_user.depth = parent_depth + 1

// مثال:
seed.depth = 0
کاربران سطح 1: depth = 1
کاربران سطح 2: depth = 2
...
کاربران سطح 7: depth = 7
```

**کد مرجع:**
```python
# در server.py خط 1855-1858
parent = await db.users.find_one({"user_id": placement_parent_id}, {"_id": 0})
parent_depth = parent.get("depth", 0) if parent else 0
new_depth = parent_depth + 1
```

### 🎯 محاسبه Level برای پاداش

**Level** برای پاداش سطحی بر اساس **تفاضل depth** محاسبه می‌شود:

```
Level = (depth کاربر جدید) - (depth کاربر دریافت‌کننده پاداش) + 1
```

**مثال:**
- کاربر A در depth=2
- کاربر B (فرزند) در depth=4 فعال می‌شود
- depth_diff = 4 - 2 = 2
- این یعنی کاربر B در **Level 3** نسبت به A قرار دارد
- پس کاربر A پاداش **level_3** را دریافت می‌کند

---

## 3️⃣ الگوریتم زیرمجموعه‌گیری و جایگذاری

### 🔍 الگوریتم BFS (Breadth-First Search)

سیستم از دو نسخه الگوریتم BFS استفاده می‌کند:

#### 🟡 نسخه Non-Atomic (سریع‌تر اما غیرایمن)

**فایل:** `placement_service.py` → `find_placement_parent()`
**فایل:** `server.py` خطوط 899-954

```python
async def find_placement_parent(referrer_id: str) -> str:
    """
    الگوریتم BFS برای پیدا کردن اولین parent با slot خالی
    
    ⚠️ هشدار: Non-atomic - برای تولید همزمان کاربران مناسب نیست
    """
    
    # 1. بررسی referrer
    referrer = await db.users.find_one({"user_id": referrer_id, "status": "active"})
    if not referrer:
        return "seed"
    
    # 2. اگر referrer کمتر از 3 فرزند دارد
    if len(referrer.get("direct_children", [])) < 3:
        return referrer_id
    
    # 3. BFS با batch queries (بهینه‌سازی شده)
    queue = deque([referrer_id])
    visited = set([referrer_id])
    batch_size = 50  # هر بار 50 کاربر fetch می‌شود
    
    while queue:
        # پردازش batch
        batch = [queue.popleft() for _ in range(min(batch_size, len(queue)))]
        
        # Fetch همه کاربران batch با یک query
        users = await db.users.find(
            {"user_id": {"$in": batch}, "status": "active"},
            {"_id": 0, "user_id": 1, "direct_children": 1}
        ).to_list(batch_size)
        
        for current_user in users:
            children = current_user.get("direct_children", [])
            
            # اگر کمتر از 3 فرزند دارد → parent پیدا شد!
            if len(children) < 3:
                return current_user["user_id"]
            
            # فرزندان را به صف اضافه کن
            for child_id in children:
                if child_id not in visited:
                    visited.add(child_id)
                    queue.append(child_id)
    
    return "seed"  # اگر جایی پیدا نشد
```

**🎯 بهینه‌سازی‌ها:**
- ✅ **Batch Queries**: به جای فچ تک‌تک، 50 کاربر یکجا fetch می‌شود
- ✅ **Visited Set**: از چرخه بی‌نهایت جلوگیری می‌کند
- ✅ **Projection**: فقط فیلدهای مورد نیاز (`user_id`, `direct_children`) fetch می‌شوند
- ✅ **Status Filter**: فقط کاربران `active` در نظر گرفته می‌شوند

#### 🟢 نسخه Atomic (ایمن اما کمی کندتر)

**فایل:** `placement_service.py` → `find_placement_parent_atomic()`
**فایل:** `server.py` خطوط 957-1050

```python
async def find_placement_parent_atomic(referrer_id: str, new_user_id: str) -> str:
    """
    🔒 الگوریتم BFS با Atomic Placement
    
    تضمین می‌کند هیچ دو کاربری در یک slot قرار نمی‌گیرند
    """
    
    # BFS مشابه نسخه قبل
    queue = deque([referrer_id])
    
    while queue:
        batch = [...]
        
        # بررسی هر کاربر در batch
        for candidate_id in batch:
            # 🔒 Atomic: reserve کردن slot با find_one_and_update
            result = await db.users.find_one_and_update(
                {
                    "user_id": candidate_id,
                    "status": "active",
                    "$expr": {"$lt": [{"$size": {"$ifNull": ["$direct_children", []]}}, 3]}
                },
                {"$push": {"direct_children": new_user_id}},
                return_document=True
            )
            
            if result:
                # موفق! این parent ما است
                return candidate_id
        
        # اضافه کردن فرزندان به queue...
    
    # آخرین تلاش: اتصال به seed
    # اگر seed هم پر بود: پیدا کردن هر کاربر با slot خالی
```

**🔐 مزایای Atomic:**
- ✅ **Race Condition Prevention**: دو کاربر نمی‌توانند همزمان یک slot را بگیرند
- ✅ **Consistency**: تضمین یکپارچگی داده
- ✅ **$expr Operator**: چک atomic تعداد فرزندان

---

## 4️⃣ منطق محاسبه پاداش (Reward Calculation Logic)

### 💰 انواع پاداش

سیستم دو نوع پاداش دارد:

#### 1. پاداش مستقیم (Direct Reward)
**دریافت‌کننده:** معرف (Referrer)
**زمان پرداخت:** هنگام فعال شدن زیرمجموعه مستقیم

**جدول پاداش:**
| ترتیب زیرمجموعه | مبلغ پیش‌فرض |
|-----------------|-------------|
| نفر اول         | $100        |
| نفر دوم         | $75         |
| نفر سوم         | $50         |
| نفرات بعدی     | $25         |

**کد:**
```python
# server.py خط 1108-1130
# شمارش زیرمجموعه‌های فعال قبلی (غیر از کاربر جدید)
direct_count = await db.users.count_documents({
    "referrer_id": referrer_id,
    "status": "active",
    "user_id": {"$ne": new_user_id}
})

if direct_count == 0:
    reward = settings["direct_first"]    # $100
elif direct_count == 1:
    reward = settings["direct_second"]   # $75
elif direct_count == 2:
    reward = settings["direct_third"]    # $50
else:
    reward = settings["direct_extra"]    # $25
```

#### 2. پاداش سطحی (Level Reward)
**دریافت‌کننده:** اجداد در درخت (Placement Line)
**زمان پرداخت:** هنگام فعال شدن هر کاربر در زیردرخت

**جدول پاداش:**
| سطح (Level) | depth_diff | مبلغ پیش‌فرض |
|------------|-----------|-------------|
| Level 3    | 2         | $10         |
| Level 5    | 4         | $5          |
| Level 7    | 6         | $2          |

**⚠️ نکته مهم:**
- Level 3 یعنی کسی که 2 سطح بالاتر است (depth_diff = 2)
- Level 5 یعنی کسی که 4 سطح بالاتر است (depth_diff = 4)
- Level 7 یعنی کسی که 6 سطح بالاتر است (depth_diff = 6)

**کد:**
```python
# server.py خط 1152-1238
new_user_depth = new_user.get("depth", 0)
current_user = await db.users.find_one({"user_id": placement_parent_id})

while current_user:
    if not current_user.get("blocked_level_reward") and not current_user.get("is_tree_complete"):
        current_depth = current_user.get("depth", 0)
        depth_diff = new_user_depth - current_depth
        
        # محدودیت 7 سطح
        if depth_diff > 7:
            break
        
        # پرداخت پاداش‌های مشخص
        if depth_diff == 2:
            reward = settings["level_3"]  # $10
        elif depth_diff == 4:
            reward = settings["level_5"]  # $5
        elif depth_diff == 6:
            reward = settings["level_7"]  # $2
        
        if reward:
            # اضافه کردن به total_points
            # ثبت تراکنش
    
    # حرکت به parent
    current_user = await db.users.find_one({"user_id": current_user.get("placement_parent_id")})
```

### 🔒 محافظت‌های امنیتی در پاداش

#### 1. Duplicate Reward Prevention
**مشکل:** جلوگیری از پرداخت چندباره پاداش برای یک کاربر

**راه‌حل:**
```python
# server.py خط 1075-1101
reward_lock_id = f"reward_lock_{new_user_id}_direct"

# ایجاد lock با upsert atomic
lock_result = await db.reward_locks.update_one(
    {"lock_id": reward_lock_id},
    {"$setOnInsert": {
        "lock_id": reward_lock_id,
        "new_user_id": new_user_id,
        "created_at": datetime.now(timezone.utc).isoformat()
    }},
    upsert=True
)

# اگر قبلاً وجود داشت، از ادامه جلوگیری می‌شود
if lock_result.upserted_id is None and lock_result.modified_count == 0:
    return  # پاداش قبلاً پرداخت شده
```

#### 2. Status Check
**مشکل:** فقط کاربران فعال باید پاداش بگیرند

**راه‌حل:**
```python
# server.py خط 1067-1070
new_user_check = await db.users.find_one({"user_id": new_user_id}, {"_id": 0, "status": 1})
if not new_user_check or new_user_check.get("status") != "active":
    return  # کاربر فعال نیست
```

#### 3. Accurate Count from Database
**مشکل:** شمارش دقیق تعداد زیرمجموعه‌ها (نه از آرایه حافظه)

**راه‌حل:**
```python
# server.py خط 1110-1114
# شمارش از دیتابیس (نه از آرایه direct_children)
direct_count = await db.users.count_documents({
    "referrer_id": referrer_id,
    "status": "active",
    "user_id": {"$ne": new_user_id}
})
```

### 🚫 شرایط مسدودسازی پاداش

- **`blocked_direct_reward`**: پاداش مستقیم مسدود شده
- **`blocked_level_reward`**: پاداش سطحی مسدود شده
- **`is_tree_complete`**: هرم کامل شده (دیگر پاداش نمی‌گیرد)

### 📈 بررسی تکمیل هرم

**فایل:** `reward_service.py` → `check_tree_completion()`
**فایل:** `server.py` خطوط 1241-1302

```python
async def check_tree_completion(user_id: str):
    """
    بررسی کامل بودن هرم با MongoDB Aggregation (بدون recursion)
    """
    
    # استفاده از $graphLookup برای شمارش زیرمجموعه‌ها
    pipeline = [
        {"$match": {"user_id": user_id}},
        {
            "$graphLookup": {
                "from": "users",
                "startWith": "$user_id",
                "connectFromField": "user_id",
                "connectToField": "placement_parent_id",
                "as": "descendants",
                "maxDepth": 6,  # 7 levels (0-6)
                "depthField": "level",
                "restrictSearchWithMatch": {
                    "status": "active",
                    "depth": {"$lte": user_depth + 7}
                }
            }
        },
        {
            "$project": {
                "total_descendants": {"$size": "$descendants"}
            }
        }
    ]
    
    result = await db.users.aggregate(pipeline).to_list(1)
    total_descendants = result[0].get("total_descendants", 0)
    
    # هرم کامل: 1092 نفر (6 سطح اول)
    if total_descendants >= 1092:
        await db.users.update_one(
            {"user_id": user_id},
            {"$set": {"is_tree_complete": True}}
        )
```

**🎯 بهینه‌سازی:**
- ✅ استفاده از **MongoDB $graphLookup** (بسیار سریع‌تر از recursion)
- ✅ **maxDepth=6**: محدود کردن جستجو به 7 سطح
- ✅ **Aggregation Pipeline**: پردازش در سمت دیتابیس

---

## 5️⃣ نقاط استفاده از BFS / Recursion / Query

### 🔄 BFS (Breadth-First Search)

| مورد استفاده | فایل | خطوط | Atomic؟ |
|--------------|------|------|---------|
| پیدا کردن parent برای جایگذاری | `placement_service.py` | 37-99 | ❌ No |
| پیدا کردن parent (نسخه atomic) | `placement_service.py` | 102-198 | ✅ Yes |
| پیدا کردن parent (server.py) | `server.py` | 899-954 | ❌ No |
| پیدا کردن parent atomic (server.py) | `server.py` | 957-1050 | ✅ Yes |

**⚡ بهینه‌سازی‌های BFS:**
- Batch queries (50 کاربر یکجا)
- Visited set (جلوگیری از چرخه)
- Status filtering (فقط active)
- Projection (فقط فیلدهای لازم)

### 🔁 Recursion

| مورد استفاده | فایل | خطوط | عمیق؟ | خطر؟ |
|--------------|------|------|------|-----|
| ساخت درخت نمایشی | `server.py` | 1704-1729 | 7 سطح | ⚠️ متوسط |

```python
# server.py خط 1704-1726
async def build_tree(uid: str, max_depth: int = 3) -> dict:
    """
    🔁 Recursion برای ساخت درخت برای نمایش
    
    ⚠️ هشدار: برای درخت‌های بزرگ می‌تواند کند باشد
    """
    if max_depth <= 0:
        return None
    
    user = await db.users.find_one({"user_id": uid}, {"_id": 0})
    
    children = []
    for child_id in user.get("direct_children", []):
        child_tree = await build_tree(child_id, max_depth - 1)  # 🔁 Recursive call
        if child_tree:
            children.append(child_tree)
    
    return {
        "user_id": user["user_id"],
        "email": user["email"],
        "status": user["status"],
        "depth": user["depth"],
        "total_points": user["total_points"],
        "direct_children_count": len(user.get("direct_children", [])),
        "children": children
    }
```

**⚠️ مشکلات بالقوه:**
- **N+1 Query Problem**: یک query برای هر نود
- **Call Stack Overflow**: برای درخت‌های بسیار بزرگ
- **Performance**: کند برای کاربران با زیرمجموعه زیاد

**💡 پیشنهاد بهبود:**
```python
# استفاده از MongoDB Aggregation به جای recursion
pipeline = [
    {"$match": {"user_id": root_user_id}},
    {
        "$graphLookup": {
            "from": "users",
            "startWith": "$user_id",
            "connectFromField": "user_id",
            "connectToField": "placement_parent_id",
            "as": "tree",
            "maxDepth": 7
        }
    }
]
```

### 📊 MongoDB Queries زیاد

#### 🔴 نقاط حساس Query

**1. توزیع پاداش (Reward Distribution)**
```python
# server.py خط 1053-1238
async def distribute_rewards(...):
    # Query 1: دریافت تنظیمات
    settings = await db.settings.find_one({"setting_id": "rewards"})
    
    # Query 2: چک وضعیت کاربر جدید
    new_user_check = await db.users.find_one({"user_id": new_user_id})
    
    # Query 3: چک lock برای duplicate prevention
    lock_result = await db.reward_locks.update_one(...)
    
    # Query 4: دریافت اطلاعات referrer
    referrer = await db.users.find_one({"user_id": referrer_id})
    
    # Query 5: شمارش direct children
    direct_count = await db.users.count_documents({...})
    
    # Query 6: Update امتیاز referrer
    await db.users.update_one({"user_id": referrer_id}, {"$inc": {...}})
    
    # Query 7: Insert transaction
    await db.transactions.insert_one(transaction)
    
    # Query 8-N: پیمایش ancestors (حلقه while)
    while current_user:
        current_user = await db.users.find_one({"user_id": parent_id})
        # + Update
        # + Insert transaction
```

**⚠️ مشکل:** **تا 20-30 query برای یک فعال‌سازی** (بسته به عمق درخت)

**2. فعال‌سازی کاربر (User Activation)**
```python
# server.py خط 1834-1970
async def activate_user(user_id: str, ...):
    # Query 1: پیدا کردن کاربر
    user = await db.users.find_one({"user_id": user_id})
    
    # Query 2-N: BFS برای پیدا کردن parent (10-50 query)
    placement_parent_id = await find_placement_parent(referrer_id)
    
    # Query N+1: دریافت parent برای محاسبه depth
    parent = await db.users.find_one({"user_id": placement_parent_id})
    
    # Query N+2: Insert payment
    await db.payments.insert_one(payment)
    
    # Query N+3: Update crypto_payments
    await db.crypto_payments.update_many(...)
    
    # Query N+4: Atomic update user status
    await db.users.update_one({"user_id": user_id, "status": "pending"}, ...)
    
    # Query N+5: Add to parent's children
    await db.users.update_one({"user_id": placement_parent_id}, ...)
    
    # Query N+6 onwards: توزیع پاداش (20-30 query دیگر!)
    await distribute_rewards(...)
    
    # Query N+... : بررسی tree completion برای ancestors
    while current:
        await check_tree_completion(current["user_id"])
        current = await db.users.find_one({"user_id": current.get("placement_parent_id")})
```

**⚠️ مشکل:** **تا 100+ query برای یک فعال‌سازی!**

**3. ساخت درخت (Build Tree)**
```python
# server.py خط 1704-1729
async def build_tree(uid: str, max_depth: int = 3):
    # Query برای هر نود!
    user = await db.users.find_one({"user_id": uid})
    
    for child_id in user.get("direct_children", []):
        child_tree = await build_tree(child_id, max_depth - 1)  # Recursive!
```

**⚠️ مشکل:** **N+1 Query Problem**
- برای درخت 7 سطحی کامل: **3279 query!**
- برای درخت 3 سطحی: **40 query!**

#### 🟢 نقاط بهینه‌شده Query

**1. Batch Queries در BFS**
```python
# server.py خط 933-936
# به جای N query جداگانه، یک query برای 50 کاربر
users = await db.users.find(
    {"user_id": {"$in": batch}, "status": "active"},
    {"_id": 0, "user_id": 1, "direct_children": 1}
).to_list(batch_size)
```

**2. MongoDB Aggregation برای Tree Completion**
```python
# server.py خط 1254-1276
# یک query aggregation به جای N query recursive
pipeline = [
    {"$match": {"user_id": user_id}},
    {"$graphLookup": {...}},  # شمارش همه descendants در یک query
    {"$project": {...}}
]
result = await db.users.aggregate(pipeline).to_list(1)
```

**3. Atomic Updates**
```python
# به جای read-then-update (2 query):
user = await db.users.find_one(...)
await db.users.update_one(...)

# استفاده از atomic operators (1 query):
await db.users.update_one(
    {"user_id": user_id},
    {"$inc": {"total_points": reward}}  # Atomic increment
)
```

### 📈 آمار استفاده Query

| عملیات | تعداد Query | نوع | بهینه؟ |
|--------|------------|-----|--------|
| ثبت‌نام کاربر | 3-5 | INSERT, FIND | ✅ خوب |
| فعال‌سازی کاربر | **80-120** | FIND, UPDATE, INSERT | ❌ زیاد |
| توزیع پاداش | **20-40** | FIND, UPDATE, INSERT | ⚠️ متوسط |
| جستجوی BFS (non-atomic) | **10-100** | FIND (batch) | ✅ بهینه شده |
| جستجوی BFS (atomic) | **10-100** | FIND_AND_UPDATE | ✅ بهینه شده |
| ساخت درخت 3 سطحی | **~40** | FIND (recursive) | ❌ بد |
| ساخت درخت 7 سطحی | **~3279** | FIND (recursive) | ❌ بسیار بد |
| بررسی tree completion | **1-2** | AGGREGATE | ✅ عالی |

---

## 6️⃣ نقاط قوت و ضعف سیستم

### ✅ نقاط قوت

1. **Atomic Operations**: استفاده از `find_one_and_update` برای جلوگیری از race conditions
2. **Duplicate Prevention**: سیستم قفل (`reward_locks`) برای جلوگیری از پاداش دوباره
3. **Batch Queries**: بهینه‌سازی BFS با fetch کردن 50 کاربر یکجا
4. **MongoDB Aggregation**: استفاده از `$graphLookup` برای tree completion
5. **Status Filtering**: فقط کاربران `active` در محاسبات
6. **Projection**: فقط فیلدهای مورد نیاز fetch می‌شوند
7. **Indexing**: ایجاد index روی فیلدهای کلیدی (user_id, referrer_id, placement_parent_id, status)

### ❌ نقاط ضعف و مشکلات

#### 1. N+1 Query Problem در Build Tree
**مشکل:**
```python
# یک query برای هر نود درخت
async def build_tree(uid: str, max_depth: int = 3):
    user = await db.users.find_one({"user_id": uid})  # Query 1
    for child_id in user.get("direct_children", []):
        child_tree = await build_tree(child_id, max_depth - 1)  # Query 2, 3, 4, ...
```

**راه‌حل پیشنهادی:**
```python
# استفاده از MongoDB Aggregation
pipeline = [
    {"$match": {"user_id": root_user_id}},
    {
        "$graphLookup": {
            "from": "users",
            "startWith": "$user_id",
            "connectFromField": "user_id",
            "connectToField": "placement_parent_id",
            "as": "descendants",
            "maxDepth": 7,
            "depthField": "level"
        }
    }
]
# سپس ساخت درخت در حافظه از نتیجه aggregation
```

#### 2. Query زیاد در Reward Distribution
**مشکل:**
- توزیع پاداش سطحی نیاز به پیمایش ancestors دارد
- هر ancestor: 1 query برای find + 1 query برای update + 1 query برای insert transaction
- برای depth=7: حداقل **21 query!**

**راه‌حل پیشنهادی:**
```python
# 1. Batch update تراکنش‌ها
transactions_to_insert = []
updates_to_apply = []

while current_user:
    if should_receive_reward:
        updates_to_apply.append({
            "user_id": current_user["user_id"],
            "amount": reward
        })
        transactions_to_insert.append({...})

# Bulk operations
await db.transactions.insert_many(transactions_to_insert)
# برای updates: استفاده از bulk_write یا aggregation pipeline
```

```python
# 2. ذخیره path در user document
# هنگام فعال‌سازی:
user.ancestor_path = [parent_id, grandparent_id, ...]
# توزیع پاداش:
ancestors = await db.users.find({"user_id": {"$in": user.ancestor_path}})
```

#### 3. Inconsistency در محاسبه Tree Complete
**مشکل:**
```python
# کد از عدد 1092 استفاده می‌کند (فقط 6 سطح)
if total_descendants >= 1092:
    is_tree_complete = True
```
اما درخت **7 سطحی** است، پس باید **3279** باشد!

**راه‌حل:**
```python
# محاسبه صحیح:
# 3 + 9 + 27 + 81 + 243 + 729 + 2187 = 3279
if total_descendants >= 3279:
    is_tree_complete = True
```

#### 4. عدم استفاده از Transaction
**مشکل:**
- فعال‌سازی کاربر شامل چندین عملیات است:
  - Update user status
  - Add to parent's children
  - Insert payment
  - Distribute rewards
  
اگر یکی fail کند، داده inconsistent می‌شود.

**راه‌حل پیشنهادی:**
```python
# استفاده از MongoDB Transactions
async with await db.client.start_session() as session:
    async with session.start_transaction():
        # تمام عملیات فعال‌سازی
        await db.users.update_one({...}, session=session)
        await db.users.update_one({...}, session=session)
        await db.payments.insert_one({...}, session=session)
        # اگر خطا: rollback خودکار
```

#### 5. Memory Overhead در BFS
**مشکل:**
```python
visited = set([referrer_id])  # ذخیره همه user_id های visited
queue = deque([referrer_id])  # صف BFS
```
برای درخت بزرگ (میلیون‌ها کاربر): مصرف حافظه زیاد

**راه‌حل:**
- استفاده از Redis برای visited set (به جای حافظه)
- محدود کردن عمق جستجو
- استفاده از database cursor pagination

#### 6. Lack of Caching
**مشکل:**
- Settings همیشه از database fetch می‌شود
- User lookups تکراری
- Tree building بدون cache

**راه‌حل پیشنهادی:**
```python
# 1. Cache settings (به جای query هر بار)
@cached(ttl=300)  # 5 minutes
async def get_reward_settings():
    return await db.settings.find_one({"setting_id": "rewards"})

# 2. Cache user data
@cached(ttl=60, key="user:{user_id}")
async def get_user(user_id: str):
    return await db.users.find_one({"user_id": user_id})

# 3. Cache tree structure
@cached(ttl=120, key="tree:{user_id}:{depth}")
async def build_tree(user_id: str, max_depth: int):
    ...
```

**نکته:** کد از `cache_manager.py` استفاده می‌کند اما فعالسازی نشده:
```python
from cache_manager import cache, cached

# اما در کد استفاده نشده!
```

---

## 7️⃣ توصیه‌های بهبود عملکرد

### 🚀 اولویت بالا

1. **حذف Recursion در Build Tree**
   ```python
   # به جای recursion، استفاده از $graphLookup
   pipeline = [{"$graphLookup": {...}}]
   ```

2. **Batch Operations در Reward Distribution**
   ```python
   # جمع‌آوری همه updates و insert در یک batch
   await db.transactions.insert_many(transactions)
   await db.users.bulk_write(updates)
   ```

3. **فعال‌سازی Cache Manager**
   ```python
   @cached(ttl=300)
   async def get_reward_settings():
       return await db.settings.find_one({"setting_id": "rewards"})
   ```

4. **اصلاح عدد Tree Completion**
   ```python
   # از 1092 به 3279 تغییر دهید
   if total_descendants >= 3279:
       is_tree_complete = True
   ```

### ⚡ اولویت متوسط

5. **استفاده از MongoDB Transactions**
   ```python
   async with session.start_transaction():
       # همه عملیات atomic
   ```

6. **ذخیره Ancestor Path**
   ```python
   user.ancestor_path = [parent1, parent2, ...]
   # سرعت بخشیدن به reward distribution
   ```

7. **Pagination در BFS**
   ```python
   # به جای load کردن همه، pagination
   cursor = db.users.find({...}).limit(100)
   ```

### 🔧 اولویت پایین

8. **Connection Pooling**
   ```python
   client = AsyncIOMotorClient(mongo_url, maxPoolSize=50)
   ```

9. **Background Jobs برای Tree Completion**
   ```python
   # به جای check در هر activation، background job
   # هر 1 ساعت یک بار check کردن
   ```

10. **Monitoring و Metrics**
    ```python
    # اضافه کردن Prometheus metrics
    query_counter.inc()
    query_duration.observe(duration)
    ```

---

## 8️⃣ خلاصه تصویری

```
┌─────────────────────────────────────────────────────────────┐
│                   SEED (admin@mlm.com)                      │
│                   depth=0, referrer=None                    │
└───────────────────────┬─────────────────────────────────────┘
                        │
        ┌───────────────┼───────────────┐
        │               │               │
    ┌───▼───┐       ┌───▼───┐       ┌───▼───┐
    │ User1 │       │ User2 │       │ User3 │
    │ d=1   │       │ d=1   │       │ d=1   │
    └───┬───┘       └───┬───┘       └───┬───┘
        │               │               │
    ┌───┼───┐       ┌───┼───┐       ┌───┼───┐
    │   │   │       │   │   │       │   │   │
  User4 5  6      User7 8  9     User10 11 12
  d=2  d=2 d=2    d=2  d=2 d=2   d=2   d=2 d=2

🎯 پاداش مستقیم (Direct):
  User1 معرفی User4 → User1 دریافت $100 (اولین زیرمجموعه)
  User1 معرفی User5 → User1 دریافت $75  (دومین زیرمجموعه)
  User1 معرفی User6 → User1 دریافت $50  (سومین زیرمجموعه)
  User1 معرفی User13→ User1 دریافت $25  (چهارمین و بعدی)

💰 پاداش سطحی (Level):
  User10 فعال شد (depth=2):
    - User3 (depth=1): depth_diff=1 → ❌ پاداشی ندارد
    - SEED (depth=0):  depth_diff=2 → ✅ دریافت $10 (Level 3)
  
  اگر User50 در depth=5 فعال شود:
    - Parent در depth=4: depth_diff=1 → ❌
    - Parent در depth=3: depth_diff=2 → ✅ $10 (Level 3)
    - Parent در depth=2: depth_diff=3 → ❌
    - Parent در depth=1: depth_diff=4 → ✅ $5 (Level 5)
    - Parent در depth=0: depth_diff=5 → ❌
```

---

## 📊 نتیجه‌گیری نهایی

### سیستم فعلی:

**✅ نقاط قوت:**
- ساختار درختی واضح و منطقی
- استفاده از Atomic Operations برای جلوگیری از race condition
- بهینه‌سازی BFS با batch queries
- استفاده از MongoDB Aggregation در بخش tree completion

**⚠️ نقاط قابل بهبود:**
- تعداد Query بسیار زیاد (80-120 query برای یک فعال‌سازی)
- Recursion در build_tree (N+1 problem)
- عدم استفاده از Cache
- عدم استفاده از MongoDB Transactions
- Inconsistency در محاسبه tree completion (1092 vs 3279)

**🎯 اولویت‌های بهبود:**
1. حذف recursion در build_tree
2. Batch operations در reward distribution
3. فعال‌سازی cache manager
4. اصلاح عدد tree completion
5. استفاده از MongoDB transactions

---

## 📚 منابع و مستندات

- **کد اصلی:**
  - `backend/server.py`: خطوط 794-1302 (الگوریتم‌های اصلی)
  - `backend/services/reward_service.py`: سیستم پاداش
  - `backend/services/placement_service.py`: الگوریتم جایگذاری
  
- **مدل‌های داده:**
  - `backend/models/user.py`: ساختار User
  - `backend/models/reward.py`: ساختار Reward و Transaction

- **معماری:**
  - `backend/ARCHITECTURE.md`: نقشه کامل پروژه

---

تاریخ تحلیل: 2025-12-17
نسخه: 1.0
تحلیل‌گر: E2 AI Agent
