"""
Bot تلگرام - نسخه ساده و مستقیم
فقط و فقط از دیتابیس می‌خواند
"""

import asyncio
import logging
import sys
sys.path.insert(0, '/app/backend')

from telegram import Update, WebAppInfo, InlineKeyboardButton, InlineKeyboardMarkup
from telegram.ext import Application, CommandHandler, ContextTypes
from telegram_settings import get_settings

# Logging
logging.basicConfig(
    format='%(asctime)s - %(levelname)s - %(message)s',
    level=logging.INFO
)
logger = logging.getLogger(__name__)


async def start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """
Handler برای /start
هر بار تنظیمات تازه از دیتابیس می‌خواند
    """
    user = update.effective_user
    logger.info(f"📩 /start از کاربر {user.id} (@{user.username})")
    
    # ذخیره chat_id در دیتابیس برای broadcast
    try:
        from motor.motor_asyncio import AsyncIOMotorClient
        import os
        from datetime import datetime, timezone
        
        client = AsyncIOMotorClient(os.getenv('MONGO_URL', 'mongodb://localhost:27017'))
        db = client[os.getenv('DB_NAME', 'mlm_database')]
        
        # ذخیره یا آپدیت chat_id
        await db.telegram_chats.update_one(
            {"chat_id": user.id},
            {
                "$set": {
                    "chat_id": user.id,
                    "username": user.username,
                    "first_name": user.first_name,
                    "last_name": user.last_name,
                    "last_interaction": datetime.now(timezone.utc).isoformat()
                }
            },
            upsert=True
        )
        logger.info(f"💾 chat_id {user.id} ذخیره شد")
        client.close()
    except Exception as e:
        logger.error(f"❌ خطا در ذخیره chat_id: {e}")
    
    # خواندن تنظیمات تازه
    settings = await get_settings()
    
    # چک enabled
    if not settings.get("enabled"):
        logger.warning(f"⚠️ Bot غیرفعال است")
        await update.message.reply_text("⚠️ ربات در حال حاضر غیرفعال است.")
        return
    
    # دریافت URL و پیام‌ها
    webapp_url = settings.get("webapp_url", "")
    welcome_msg = settings.get("welcome_message", "")
    welcome_ref_msg = settings.get("welcome_with_ref_message", "")
    
    if not webapp_url:
        logger.error("❌ webapp_url خالی است")
        return
    
    # چک کردن referral code
    referral_code = None
    if context.args and len(context.args) > 0:
        referral_code = context.args[0]
    
    # ایجاد یک session_id منحصر به فرد برای ارتباط با ثبت‌نام
    import uuid
    session_id = str(uuid.uuid4())
    
    # ذخیره session_id با chat_id
    try:
        client = AsyncIOMotorClient(os.getenv('MONGO_URL', 'mongodb://localhost:27017'))
        db = client[os.getenv('DB_NAME', 'mlm_database')]
        
        await db.telegram_sessions.update_one(
            {"session_id": session_id},
            {
                "$set": {
                    "session_id": session_id,
                    "chat_id": user.id,
                    "created_at": datetime.now(timezone.utc).isoformat(),
                    "used": False
                }
            },
            upsert=True
        )
        client.close()
    except Exception as e:
        logger.error(f"❌ خطا در ذخیره session: {e}")
    
    # ساخت URL و پیام
    if referral_code:
        url = f"{webapp_url}/register?ref={referral_code}&source=telegram&tg_session={session_id}"
        message = welcome_ref_msg.format(
            user_name=user.first_name,
            referral_code=referral_code
        ) if welcome_ref_msg else f"🎉 سلام {user.first_name}!"
        logger.info(f"🔗 با رفرال: {referral_code}")
    else:
        url = f"{webapp_url}?source=telegram&tg_session={session_id}"
        message = welcome_msg.format(
            user_name=user.first_name
        ) if welcome_msg else f"👋 سلام {user.first_name}!"
        logger.info("🔗 بدون رفرال")
    
    # ساخت دکمه
    keyboard = [
        [InlineKeyboardButton("🚀 ورود به سیستم", web_app=WebAppInfo(url=url))]
    ]
    reply_markup = InlineKeyboardMarkup(keyboard)
    
    # ارسال پیام
    await update.message.reply_text(
        message,
        reply_markup=reply_markup,
        parse_mode='Markdown'
    )
    
    logger.info(f"✅ پیام ارسال شد ({len(message)} حرف)")


async def help_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """Help command"""
    await update.message.reply_text("📚 برای شروع، دستور /start را بفرستید")


async def send_activation_message(telegram_user_id: int, user_email: str, referral_code: str, bot_username: str, webapp_url: str):
    """
    ارسال پیام فعالسازی به کاربر
    شامل لینک‌های دعوت (سایت و تلگرام)
    """
    try:
        settings = await get_settings()
        token = settings.get("bot_token", "")
        
        if not token:
            logger.error("❌ Token برای ارسال پیام فعالسازی خالی است")
            return False
        
        # ساخت لینک‌های دعوت
        website_referral_link = f"{webapp_url}/register?ref={referral_code}"
        telegram_referral_link = f"https://t.me/{bot_username}?start={referral_code}"
        
        # پیام فعالسازی
        message = f"""🎉 تبریک {user_email} عزیز!

✅ حساب شما با موفقیت فعال شد!

🔗 لینک‌های دعوت شما:

🌐 **لینک دعوت (وب):**
{website_referral_link}

✈️ **لینک دعوت (تلگرام):**
{telegram_referral_link}

💰 با دعوت دوستان خود، پاداش دریافت کنید!

🚀 برای مشاهده داشبورد خود، روی دکمه زیر کلیک کنید."""

        # ساخت دکمه
        keyboard = [
            [InlineKeyboardButton("📊 داشبورد من", web_app=WebAppInfo(url=webapp_url))]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        # ارسال پیام
        from telegram import Bot
        bot = Bot(token)
        await bot.send_message(
            chat_id=telegram_user_id,
            text=message,
            reply_markup=reply_markup,
            parse_mode='Markdown'
        )
        
        logger.info(f"✅ پیام فعالسازی به {telegram_user_id} ارسال شد")
        return True
        
    except Exception as e:
        logger.error(f"❌ خطا در ارسال پیام فعالسازی: {e}")
        return False


async def send_broadcast_message(user_ids: list, message: str, bot_token: str):
    """
    ارسال پیام عمومی به لیست کاربران
    """
    from telegram import Bot
    
    bot = Bot(bot_token)
    success_count = 0
    failed_count = 0
    
    for user_id in user_ids:
        try:
            await bot.send_message(
                chat_id=user_id,
                text=message,
                parse_mode='Markdown'
            )
            success_count += 1
            logger.info(f"✅ پیام به {user_id} ارسال شد")
            await asyncio.sleep(0.1)  # جلوگیری از rate limit
        except Exception as e:
            failed_count += 1
            logger.error(f"❌ خطا در ارسال به {user_id}: {e}")
    
    return {"success": success_count, "failed": failed_count}


def main():
    """راه‌اندازی Bot"""
    logger.info("🤖 شروع Bot...")
    
    # خواندن تنظیمات
    loop = asyncio.new_event_loop()
    asyncio.set_event_loop(loop)
    settings = loop.run_until_complete(get_settings())
    
    if not settings.get("enabled"):
        logger.error("❌ Bot غیرفعال است!")
        loop.close()
        return
    
    token = settings.get("bot_token", "")
    if not token:
        logger.error("❌ Token خالی است!")
        loop.close()
        return
    
    # نمایش تنظیمات
    logger.info("="*50)
    logger.info("✅ تنظیمات:")
    logger.info(f"   Enabled: {settings.get('enabled')}")
    logger.info(f"   Username: @{settings.get('bot_username')}")
    logger.info(f"   Token: {token[:20]}...")
    logger.info(f"   پیام عادی: {len(settings.get('welcome_message', ''))} حرف")
    logger.info(f"   پیام رفرال: {len(settings.get('welcome_with_ref_message', ''))} حرف")
    logger.info("="*50)
    
    # ساخت Application
    application = Application.builder().token(token).build()
    
    # Handlers
    application.add_handler(CommandHandler("start", start))
    application.add_handler(CommandHandler("help", help_command))
    
    # شروع
    logger.info("✅ Bot در حال اجرا...")
    try:
        application.run_polling(allowed_updates=Update.ALL_TYPES)
    except KeyboardInterrupt:
        logger.info("🛑 Bot متوقف شد")
    finally:
        loop.close()


if __name__ == '__main__':
    main()
