"""Placement Service - جایگذاری کاربر در درخت بدون BFS"""

from typing import Dict, Optional
import logging
from .path_manager import PathManager

logger = logging.getLogger(__name__)

# Database connection (باید از بیرون set شود)
_db = None


def set_database(db):
    """تنظیم database connection"""
    global _db
    _db = db


class PlacementService:
    """سرویس جایگذاری کاربر - بدون BFS"""
    
    @staticmethod
    async def find_available_parent(referrer_id: str) -> Optional[Dict]:
        """
        پیدا کردن parent با slot خالی - بدون BFS!
        
        Strategy:
        1. اول سعی کن زیر referrer قرار بگیرد
        2. اگر referrer پر بود، اولین parent با slot خالی
        
        Args:
            referrer_id: شناسه معرف
            
        Returns:
            اطلاعات parent یا None
            {
                "parent_id": str,
                "parent_path": str,
                "parent_depth": int,
                "position": int  # 0, 1, or 2
            }
        """
        
        # Strategy 1: Check if referrer has space
        referrer = await _db.users.find_one(
            {
                "user_id": referrer_id,
                "status": "active",
                "direct_children_count": {"$lt": 3}
            },
            {"_id": 0, "user_id": 1, "placement_path": 1, "depth": 1, "direct_children_count": 1}
        )
        
        if referrer:
            logger.info(f"✅ Referrer {referrer_id} has space")
            return {
                "parent_id": referrer["user_id"],
                "parent_path": referrer.get("placement_path"),
                "parent_depth": referrer.get("depth", 0),
                "position": referrer.get("direct_children_count", 0)
            }
        
        # Strategy 2: Find first available parent (با index سریع است)
        # Sort by depth: BFS-like بدون BFS!
        parent = await _db.users.find_one(
            {
                "status": "active",
                "direct_children_count": {"$lt": 3}
            },
            {"_id": 0, "user_id": 1, "placement_path": 1, "depth": 1, "direct_children_count": 1},
            sort=[("depth", 1)]  # کمترین depth اول (BFS behavior)
        )
        
        if parent:
            logger.info(f"✅ Found available parent: {parent['user_id']} at depth {parent['depth']}")
            return {
                "parent_id": parent["user_id"],
                "parent_path": parent.get("placement_path"),
                "parent_depth": parent.get("depth", 0),
                "position": parent.get("direct_children_count", 0)
            }
        
        logger.error("❌ No available parent found!")
        return None
    
    @staticmethod
    async def place_user_in_tree(user_id: str, parent_info: Dict) -> Dict:
        """
        محاسبه اطلاعات placement برای کاربر - بدون query
        
        Args:
            user_id: شناسه کاربر جدید
            parent_info: اطلاعات parent از find_available_parent
            
        Returns:
            اطلاعات placement
            {
                "placement_path": str,
                "placement_parent_id": str,
                "placement_position": int,
                "depth": int,
                "ancestor_ids": List[str]
            }
        """
        parent_id = parent_info["parent_id"]
        parent_path = parent_info["parent_path"]
        parent_depth = parent_info["parent_depth"]
        position = parent_info["position"]
        
        # ساخت path جدید (محاسبه ساده - بدون query)
        new_path = PathManager.build_child_path(parent_path, user_id)
        
        # محاسبه depth (محاسبه ساده - بدون query)
        new_depth = parent_depth + 1
        
        # استخراج ancestors (محاسبه ساده - بدون query)
        ancestor_ids = PathManager.extract_ancestors(new_path)
        
        logger.info(f"📍 Placed {user_id} under {parent_id}: path={new_path}, depth={new_depth}")
        
        return {
            "placement_path": new_path,
            "placement_parent_id": parent_id,
            "placement_position": position,
            "depth": new_depth,
            "ancestor_ids": ancestor_ids
        }
    
    @staticmethod
    async def reserve_slot_atomic(parent_id: str, child_id: str, position: int) -> bool:
        """
        Reserve کردن atomic یک slot در parent
        
        Args:
            parent_id: شناسه parent
            child_id: شناسه child جدید
            position: موقعیت (0, 1, 2)
            
        Returns:
            True اگر موفق باشد
        """
        result = await _db.users.find_one_and_update(
            {
                "user_id": parent_id,
                "status": "active",
                "direct_children_count": {"$lt": 3}
            },
            {
                "$push": {"direct_children": child_id},
                "$inc": {"direct_children_count": 1}
            },
            return_document=True
        )
        
        return result is not None
