import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Tabs, TabsContent, TabsList, TabsTrigger } from './ui/tabs';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from './ui/dialog';
import { FileText, Wallet, TrendingDown, Activity } from 'lucide-react';

const API = process.env.REACT_APP_BACKEND_URL;

export default function UserReport({ userId, userEmail }) {
  const [report, setReport] = useState(null);
  const [loading, setLoading] = useState(false);

  const getAuthHeaders = () => ({
    Authorization: `Bearer ${localStorage.getItem('token')}`
  });

  const fetchReport = async () => {
    if (!userId) return;
    
    setLoading(true);
    try {
      const headers = getAuthHeaders();
      const response = await axios.get(`${API}/api/admin/users/${userId}/report`, { headers });
      setReport(response.data);
    } catch (error) {
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (userId) {
      fetchReport();
    }
  }, [userId]);

  if (loading) {
    return <div className="text-center py-8">در حال بارگذاری...</div>;
  }

  if (!report) {
    return null;
  }

  return (
    <div className="space-y-4">
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <FileText className="w-5 h-5" />
            گزارش کامل کاربر: {userEmail}
          </CardTitle>
        </CardHeader>
        <CardContent>
          <Tabs defaultValue="wallet">
            <TabsList className="grid w-full grid-cols-4">
              <TabsTrigger value="wallet">
                <Wallet className="w-4 h-4 ml-1" />
                کیف پول
              </TabsTrigger>
              <TabsTrigger value="withdrawals">
                <TrendingDown className="w-4 h-4 ml-1" />
                برداشت‌ها
              </TabsTrigger>
              <TabsTrigger value="stats">
                <Activity className="w-4 h-4 ml-1" />
                آمار
              </TabsTrigger>
              <TabsTrigger value="transactions">
                تراکنش‌ها
              </TabsTrigger>
            </TabsList>

            {/* کیف پول */}
            <TabsContent value="wallet" className="space-y-3">
              <div className="bg-gradient-to-r from-blue-50 to-indigo-50 p-4 rounded-lg border-2 border-blue-200">
                <h3 className="font-semibold mb-2">📍 اطلاعات کیف پول</h3>
                <div className="space-y-2 text-sm">
                  <div>
                    <span className="text-gray-600">آدرس:</span>
                    <div className="font-mono bg-white p-2 rounded mt-1 break-all">
                      {report.wallet_info?.address || 'ثبت نشده'}
                    </div>
                  </div>
                  <div>
                    <span className="text-gray-600">شبکه:</span>
                    <span className="font-semibold mr-2">{report.wallet_info?.network || 'N/A'}</span>
                  </div>
                </div>
              </div>

              {/* پرداخت‌های کریپتو */}
              <h4 className="font-semibold mt-4">تاریخچه پرداخت‌های کریپتو:</h4>
              {report.crypto_payments?.length > 0 ? (
                <div className="space-y-2">
                  {report.crypto_payments.map((payment) => (
                    <div key={payment.payment_id} className="border rounded p-3 bg-white">
                      <div className="flex justify-between items-start">
                        <div>
                          <div className="font-semibold">{payment.amount} USDT</div>
                          <div className="text-xs text-gray-500">{payment.network}</div>
                          <div className="text-xs text-gray-400 mt-1">
                            {new Date(payment.created_at).toLocaleString('fa-IR')}
                          </div>
                        </div>
                        <span className={`px-2 py-1 rounded text-xs ${
                          payment.status === 'completed' 
                            ? 'bg-green-100 text-green-700' 
                            : 'bg-amber-100 text-amber-700'
                        }`}>
                          {payment.status === 'completed' ? 'تکمیل' : 'در انتظار'}
                        </span>
                      </div>
                    </div>
                  ))}
                </div>
              ) : (
                <p className="text-gray-500 text-sm">پرداختی ثبت نشده</p>
              )}
            </TabsContent>

            {/* برداشت‌ها */}
            <TabsContent value="withdrawals" className="space-y-3">
              {report.withdrawals?.length > 0 ? (
                report.withdrawals.map((w) => (
                  <div key={w.withdrawal_id} className="border-2 rounded-lg p-4 bg-white">
                    <div className="flex justify-between items-start mb-2">
                      <div>
                        <div className="text-xl font-bold text-blue-600">{w.amount} USDT</div>
                        <div className="text-xs text-gray-500">{w.network}</div>
                        <div className="font-mono text-xs mt-2 bg-gray-100 p-2 rounded break-all">
                          {w.crypto_address}
                        </div>
                        <div className="text-xs text-gray-400 mt-2">
                          درخواست: {new Date(w.requested_at).toLocaleDateString('fa-IR')}
                        </div>
                      </div>
                      <span className={`px-3 py-1 rounded-full text-sm font-semibold ${
                        w.status === 'completed' ? 'bg-green-100 text-green-700' :
                        w.status === 'rejected' ? 'bg-red-100 text-red-700' :
                        'bg-amber-100 text-amber-700'
                      }`}>
                        {w.status === 'completed' ? '✓ تکمیل' : 
                         w.status === 'rejected' ? '✗ رد' : 
                         '⏳ در انتظار'}
                      </span>
                    </div>
                    
                    {w.transaction_hash && (
                      <div className="mt-2 p-2 bg-green-50 rounded text-xs">
                        <strong>Hash:</strong> <span className="font-mono">{w.transaction_hash}</span>
                      </div>
                    )}
                    
                    {w.rejection_reason && (
                      <div className="mt-2 p-2 bg-red-50 rounded text-xs text-red-700">
                        <strong>دلیل رد:</strong> {w.rejection_reason}
                      </div>
                    )}
                  </div>
                ))
              ) : (
                <p className="text-gray-500 text-center py-8">درخواست برداشتی ثبت نشده</p>
              )}
            </TabsContent>

            {/* آمار */}
            <TabsContent value="stats">
              <div className="grid grid-cols-2 gap-4">
                <div className="bg-blue-50 p-4 rounded-lg border-2 border-blue-200">
                  <div className="text-sm text-gray-600">امتیاز فعلی</div>
                  <div className="text-2xl font-bold text-blue-600">
                    {report.stats?.total_points?.toFixed(2)} USDT
                  </div>
                </div>
                
                <div className="bg-green-50 p-4 rounded-lg border-2 border-green-200">
                  <div className="text-sm text-gray-600">مجموع برداشت</div>
                  <div className="text-2xl font-bold text-green-600">
                    {report.stats?.total_withdrawn?.toFixed(2)} USDT
                  </div>
                </div>
                
                <div className="bg-amber-50 p-4 rounded-lg border-2 border-amber-200">
                  <div className="text-sm text-gray-600">در انتظار برداشت</div>
                  <div className="text-2xl font-bold text-amber-600">
                    {report.stats?.pending_withdrawals?.toFixed(2)} USDT
                  </div>
                </div>
                
                <div className="bg-purple-50 p-4 rounded-lg border-2 border-purple-200">
                  <div className="text-sm text-gray-600">تعداد برداشت</div>
                  <div className="text-2xl font-bold text-purple-600">
                    {report.stats?.withdrawal_count}
                  </div>
                </div>
              </div>
            </TabsContent>

            {/* تراکنش‌های اخیر */}
            <TabsContent value="transactions" className="space-y-2">
              {report.recent_transactions?.length > 0 ? (
                report.recent_transactions.map((tx, idx) => (
                  <div key={idx} className="border rounded p-3 bg-white text-sm">
                    <div className="flex justify-between items-start">
                      <div>
                        <div className="font-semibold">{tx.type}</div>
                        <div className="text-xs text-gray-500">{tx.description}</div>
                        <div className="text-xs text-gray-400 mt-1">
                          {new Date(tx.timestamp).toLocaleString('fa-IR')}
                        </div>
                      </div>
                      <div className={`font-bold ${tx.amount > 0 ? 'text-green-600' : 'text-red-600'}`}>
                        {tx.amount > 0 ? '+' : ''}{tx.amount} USDT
                      </div>
                    </div>
                  </div>
                ))
              ) : (
                <p className="text-gray-500 text-center py-8">تراکنشی ثبت نشده</p>
              )}
            </TabsContent>
          </Tabs>
        </CardContent>
      </Card>
    </div>
  );
}
