"""User Activation Service - فعال‌سازی کاربر با Materialized Path"""

from typing import Dict
import uuid
import logging
from datetime import datetime, timezone
from ..tree.path_manager import PathManager
from ..tree.placement import PlacementService
from ..tree.tree_stats import TreeStatsService
from ..rewards.distributor import RewardDistributor

logger = logging.getLogger(__name__)

_db = None

def set_database(db):
    """تنظیم database connection"""
    global _db
    _db = db


class UserActivationService:
    """سرویس فعال‌سازی کاربر - با Materialized Path"""
    
    @staticmethod
    async def activate_user(user_id: str, admin_user_id: str = None) -> Dict:
        """
        فعال‌سازی کامل کاربر
        
        Process:
        1. Get user (pending)
        2. Find placement parent (1-2 queries)
        3. Build path and calculate depth (0 queries - math)
        4. Get ancestors data (1 query)
        5. Build ancestor_depths (0 queries - math)
        6. Atomic update user + parent (1 query با transaction)
        7. Insert payment (1 query)
        8. Distribute rewards (8 queries)
        9. Update descendants count (1 query)
        10. Check tree completion (1 query)
        
        Total: ~16 queries (قبلی: 80-120!)
        
        Args:
            user_id: شناسه کاربر
            admin_user_id: شناسه ادمین فعال‌ساز (اختیاری)
            
        Returns:
            اطلاعات کاربر فعال شده
        """
        
        # Step 1: Get user
        user = await _db.users.find_one(
            {"user_id": user_id},
            {"_id": 0}
        )
        
        if not user:
            raise ValueError(f"کاربر {user_id} یافت نشد")
        
        if user["status"] != "pending":
            raise ValueError("کاربر قبلاً فعال شده است")
        
        referrer_id = user.get("referrer_id", "seed")
        
        # Step 2: Find placement parent
        parent_info = await PlacementService.find_available_parent(referrer_id)
        if not parent_info:
            raise ValueError("امکان یافتن جایگاه مناسب وجود ندارد")
        
        # Step 3: Build path - محاسبه ریاضی (0 queries)
        placement_info = await PlacementService.place_user_in_tree(user_id, parent_info)
        
        # Step 4: Get ancestors data (1 query)
        ancestor_ids = placement_info["ancestor_ids"]
        ancestors_data = []
        
        logger.info(f"🔍 ancestor_ids from path: {ancestor_ids}")
        
        if ancestor_ids:
            ancestors = await _db.users.find(
                {"user_id": {"$in": ancestor_ids}},
                {"_id": 0, "user_id": 1, "depth": 1}
            ).to_list(len(ancestor_ids))
            ancestors_data = ancestors
            logger.info(f"🔍 Found {len(ancestors_data)} ancestors in DB")
        else:
            logger.info(f"🔍 No ancestors (depth 1 user)")
        
        # Step 5: Build ancestor_depths - محاسبه ریاضی (0 queries)
        ancestor_depths = PathManager.build_ancestor_depths(
            placement_info["placement_path"],
            ancestors_data
        )
        
        logger.info(f"🔍 Built ancestor_depths: {ancestor_depths}")
        
        # Step 6: Get settings for payment
        settings = await _db.settings.find_one({"setting_id": "rewards"})
        activation_fee = settings.get("activation_fee", 100.0) if settings else 100.0
        
        # Step 7: Atomic update user + parent (با transaction)
        from pymongo import UpdateOne
        
        # Update user
        user_update = {
            "status": "active",
            "placement_path": placement_info["placement_path"],
            "placement_parent_id": placement_info["placement_parent_id"],
            "placement_position": placement_info["placement_position"],
            "depth": placement_info["depth"],
            "ancestor_depths": ancestor_depths,
            "direct_children_count": 0,  # شروع با 0
            "total_descendants": 0,       # شروع با 0
            "activated_at": datetime.now(timezone.utc).isoformat()
        }
        
        # Atomic: Update user status
        activation_result = await _db.users.update_one(
            {"user_id": user_id, "status": "pending"},
            {"$set": user_update}
        )
        
        if activation_result.modified_count == 0:
            raise ValueError("کاربر توسط عملیات دیگری فعال شده است")
        
        # Atomic: Reserve slot in parent
        success = await PlacementService.reserve_slot_atomic(
            placement_info["placement_parent_id"],
            user_id,
            placement_info["placement_position"]
        )
        
        if not success:
            logger.warning(f"⚠️ Parent slot reservation failed - may need retry")
        
        # Step 8: Insert payment record
        payment = {
            "payment_id": str(uuid.uuid4()),
            "user_id": user_id,
            "amount": activation_fee,
            "timestamp": datetime.now(timezone.utc).isoformat(),
            "description": f"پرداخت ورود به سیستم - {user['email']}"
        }
        await _db.payments.insert_one(payment)
        
        # Step 9: Distribute rewards (8 queries)
        try:
            await RewardDistributor.distribute_all_rewards(user_id)
        except Exception as e:
            logger.error(f"❌ Error distributing rewards: {e}")
            # ادامه می‌دهیم - فعال‌سازی موفق بوده
        
        # Step 10: Update descendants count (1 query)
        try:
            await TreeStatsService.update_descendants_count(
                placement_info["placement_path"],
                increment=1
            )
        except Exception as e:
            logger.error(f"❌ Error updating descendants: {e}")
        
        # Step 11: Check tree completion (1 query)
        try:
            await TreeStatsService.check_tree_completion(
                placement_info["placement_path"]
            )
        except Exception as e:
            logger.error(f"❌ Error checking tree completion: {e}")
        
        logger.info(f"🎉 User {user_id} activated successfully!")
        
        return {
            "user_id": user_id,
            "email": user["email"],
            "status": "active",
            "placement_path": placement_info["placement_path"],
            "placement_parent_id": placement_info["placement_parent_id"],
            "placement_position": placement_info["placement_position"],
            "depth": placement_info["depth"],
            "ancestor_depths": ancestor_depths,
            "referrer_id": user.get("referrer_id"),
            "message": "فعال‌سازی موفق"
        }
