#!/bin/bash

################################################################################
# MLM System - SSL/HTTPS Auto Setup with Zero Downtime
# تنظیم خودکار SSL با تمدید اتوماتیک
################################################################################

set -e

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

clear
echo "════════════════════════════════════════════════════════════════"
echo -e "${BLUE}🔒 MLM System - SSL/HTTPS Setup${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

# چک کردن domain
echo -e "${YELLOW}⚠️  نکات مهم قبل از شروع:${NC}"
echo "1. دامنه شما باید به IP سرور اشاره کرده باشد (DNS تنظیم شده)"
echo "2. پورت 80 و 443 باید آزاد باشد"
echo "3. سیستم برای چند ثانیه restart می‌شود (بدون قطعی طولانی)"
echo ""

read -p "آیا دامنه شما به این سرور اشاره می‌کند؟ (y/n): " DNS_READY
if [ "$DNS_READY" != "y" ]; then
    echo -e "${RED}❌ ابتدا DNS را تنظیم کنید و سپس مجدد اجرا کنید${NC}"
    exit 1
fi

echo ""
read -p "دامنه خود را وارد کنید (مثال: mlm.example.com): " DOMAIN
read -p "ایمیل خود را وارد کنید (برای اطلاعیه‌های Let's Encrypt): " EMAIL

if [ -z "$DOMAIN" ] || [ -z "$EMAIL" ]; then
    echo -e "${RED}❌ دامنه و ایمیل الزامی است${NC}"
    exit 1
fi

echo ""
echo -e "${GREEN}✔ دامنه: $DOMAIN${NC}"
echo -e "${GREEN}✔ ایمیل: $EMAIL${NC}"
echo ""

read -p "ادامه می‌دهیم؟ (y/n): " CONFIRM
if [ "$CONFIRM" != "y" ]; then
    echo "لغو شد"
    exit 0
fi

################################################################################
# 1. نصب Certbot
################################################################################

echo ""
echo "════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}📦 نصب Certbot...${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

if ! command -v certbot &> /dev/null; then
    sudo apt update -qq
    sudo apt install -y certbot python3-certbot-nginx
    echo -e "${GREEN}✔ Certbot نصب شد${NC}"
else
    echo -e "${GREEN}✔ Certbot موجود است${NC}"
fi

################################################################################
# 2. توقف موقت Nginx container (برای دریافت گواهی)
################################################################################

echo ""
echo "════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}🔒 دریافت گواهی SSL...${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

echo -e "${YELLOW}⏳ توقف موقت Nginx (10 ثانیه)...${NC}"
sudo docker compose stop nginx

# دریافت گواهی با standalone mode
echo -e "${YELLOW}⏳ درخواست گواهی از Let's Encrypt...${NC}"
sudo certbot certonly \
    --standalone \
    --non-interactive \
    --agree-tos \
    --email "$EMAIL" \
    -d "$DOMAIN" \
    --preferred-challenges http

if [ $? -eq 0 ]; then
    echo -e "${GREEN}✔ گواهی SSL با موفقیت دریافت شد${NC}"
else
    echo -e "${RED}❌ خطا در دریافت گواهی${NC}"
    echo -e "${YELLOW}⏳ راه‌اندازی مجدد Nginx...${NC}"
    sudo docker compose start nginx
    exit 1
fi

################################################################################
# 3. ایجاد Nginx Config با SSL
################################################################################

echo ""
echo "════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}⚙️  تنظیم Nginx برای HTTPS...${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

# Backup nginx config قدیمی
if [ -f "nginx.conf" ]; then
    cp nginx.conf nginx.conf.backup.$(date +%Y%m%d_%H%M%S)
    echo -e "${GREEN}✔ Backup از config قدیمی گرفته شد${NC}"
fi

# ایجاد nginx config جدید با SSL
cat > nginx.conf << 'NGINX_EOF'
upstream backend {
    server backend:8001;
}

upstream frontend {
    server frontend:80;
}

# HTTP to HTTPS redirect
server {
    listen 80;
    server_name DOMAIN_PLACEHOLDER;
    
    # برای تمدید گواهی
    location /.well-known/acme-challenge/ {
        root /var/www/certbot;
    }
    
    # redirect به HTTPS
    location / {
        return 301 https://$server_name$request_uri;
    }
}

# HTTPS server
server {
    listen 443 ssl http2;
    server_name DOMAIN_PLACEHOLDER;
    
    # SSL certificates
    ssl_certificate /etc/letsencrypt/live/DOMAIN_PLACEHOLDER/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/DOMAIN_PLACEHOLDER/privkey.pem;
    
    # SSL optimization
    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_ciphers 'ECDHE-ECDSA-AES128-GCM-SHA256:ECDHE-RSA-AES128-GCM-SHA256:ECDHE-ECDSA-AES256-GCM-SHA384:ECDHE-RSA-AES256-GCM-SHA384';
    ssl_prefer_server_ciphers off;
    ssl_session_cache shared:SSL:10m;
    ssl_session_timeout 10m;
    
    # Security headers
    add_header Strict-Transport-Security "max-age=31536000; includeSubDomains" always;
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-Content-Type-Options "nosniff" always;
    add_header X-XSS-Protection "1; mode=block" always;
    
    client_max_body_size 50M;
    
    # Logging
    access_log /var/log/nginx/mlm_access.log;
    error_log /var/log/nginx/mlm_error.log;
    
    # Backend API
    location /api {
        proxy_pass http://backend;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_cache_bypass $http_upgrade;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        
        # Timeouts
        proxy_connect_timeout 60s;
        proxy_send_timeout 60s;
        proxy_read_timeout 60s;
    }
    
    # Health check
    location /health {
        proxy_pass http://backend/api/health;
        access_log off;
    }
    
    # Frontend - Everything else
    location / {
        proxy_pass http://frontend;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_cache_bypass $http_upgrade;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
}
NGINX_EOF

# جایگذاری domain در config
sed -i "s/DOMAIN_PLACEHOLDER/$DOMAIN/g" nginx.conf

echo -e "${GREEN}✔ Nginx config برای HTTPS آماده شد${NC}"

################################################################################
# 4. به‌روزرسانی docker-compose.yml
################################################################################

echo ""
echo -e "${YELLOW}⚙️  به‌روزرسانی docker-compose.yml...${NC}"

# Backup docker-compose.yml قدیمی
if [ -f "docker-compose.yml" ]; then
    cp docker-compose.yml docker-compose.yml.backup.$(date +%Y%m%d_%H%M%S)
    echo -e "${GREEN}✔ Backup از docker-compose.yml گرفته شد${NC}"
fi

# اضافه کردن SSL volumes به nginx service
# استفاده از Python برای ویرایش YAML
python3 << 'PYTHON_EOF'
import yaml
import sys

try:
    with open('docker-compose.yml', 'r') as f:
        compose = yaml.safe_load(f)
    
    # اضافه کردن پورت 443 و volumes
    if 'services' in compose and 'nginx' in compose['services']:
        nginx = compose['services']['nginx']
        
        # Ports
        if 'ports' not in nginx:
            nginx['ports'] = []
        if '443:443' not in str(nginx['ports']):
            nginx['ports'].append('443:443')
        
        # Volumes
        if 'volumes' not in nginx:
            nginx['volumes'] = []
        
        ssl_volumes = [
            '/etc/letsencrypt:/etc/letsencrypt:ro',
            '/var/lib/letsencrypt:/var/lib/letsencrypt:ro'
        ]
        
        for vol in ssl_volumes:
            if vol not in nginx['volumes']:
                nginx['volumes'].append(vol)
    
    with open('docker-compose.yml', 'w') as f:
        yaml.dump(compose, f, default_flow_style=False, sort_keys=False)
    
    print("SUCCESS")
except Exception as e:
    print(f"ERROR: {e}")
    sys.exit(1)
PYTHON_EOF

if [ $? -eq 0 ]; then
    echo -e "${GREEN}✔ docker-compose.yml به‌روزرسانی شد${NC}"
else
    echo -e "${RED}❌ خطا در به‌روزرسانی docker-compose.yml${NC}"
    echo -e "${YELLOW}⚠️  دستی باید volumes اضافه کنید${NC}"
fi

################################################################################
# 5. به‌روزرسانی Frontend .env
################################################################################

echo ""
echo -e "${YELLOW}⚙️  به‌روزرسانی Frontend config...${NC}"

if [ -f "frontend/.env" ]; then
    cp frontend/.env frontend/.env.backup.$(date +%Y%m%d_%H%M%S)
    
    # تغییر URL ها به HTTPS
    sed -i "s|http://|https://|g" frontend/.env
    sed -i "s|REACT_APP_BACKEND_URL=.*|REACT_APP_BACKEND_URL=https://$DOMAIN|g" frontend/.env
    sed -i "s|REACT_APP_API_URL=.*|REACT_APP_API_URL=https://$DOMAIN/api|g" frontend/.env
    
    echo -e "${GREEN}✔ Frontend .env به‌روزرسانی شد${NC}"
fi

################################################################################
# 6. تنظیم Auto-Renewal
################################################################################

echo ""
echo "════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}🔄 تنظیم تمدید خودکار...${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

# ایجاد renewal hook script
sudo mkdir -p /etc/letsencrypt/renewal-hooks/deploy

cat | sudo tee /etc/letsencrypt/renewal-hooks/deploy/reload-nginx.sh > /dev/null << 'HOOK_EOF'
#!/bin/bash
# Reload nginx after certificate renewal
cd /root/deployment_package || cd ~/deployment_package || cd /opt/mlm
docker compose exec nginx nginx -s reload
HOOK_EOF

sudo chmod +x /etc/letsencrypt/renewal-hooks/deploy/reload-nginx.sh

echo -e "${GREEN}✔ Renewal hook ایجاد شد${NC}"

# فعال‌سازی certbot timer
sudo systemctl enable certbot.timer
sudo systemctl start certbot.timer

echo -e "${GREEN}✔ Certbot timer فعال شد (تمدید خودکار هر 12 ساعت چک می‌شود)${NC}"

# تست renewal
echo ""
echo -e "${YELLOW}⏳ تست تمدید خودکار...${NC}"
sudo certbot renew --dry-run

if [ $? -eq 0 ]; then
    echo -e "${GREEN}✔ تست تمدید موفق بود${NC}"
else
    echo -e "${YELLOW}⚠️  تست تمدید با خطا مواجه شد (نگران نباشید، گواهی فعلی کار می‌کند)${NC}"
fi

################################################################################
# 7. راه‌اندازی مجدد سیستم
################################################################################

echo ""
echo "════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}🚀 راه‌اندازی مجدد با HTTPS...${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

echo -e "${YELLOW}⏳ Rebuild frontend با HTTPS URLs...${NC}"
cd frontend
docker build -t mlm-frontend . > /dev/null 2>&1
cd ..

echo -e "${YELLOW}⏳ راه‌اندازی مجدد تمام سرویس‌ها...${NC}"
sudo docker compose down
sleep 2
sudo docker compose up -d --build

echo ""
echo -e "${YELLOW}⏳ انتظار برای راه‌اندازی (15 ثانیه)...${NC}"
sleep 15

################################################################################
# 8. تست و نمایش اطلاعات
################################################################################

echo ""
echo "════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}🧪 تست سیستم...${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

# تست HTTPS
echo -e "${YELLOW}⏳ تست HTTPS connection...${NC}"
if curl -sSf -k https://$DOMAIN/health > /dev/null 2>&1; then
    echo -e "${GREEN}✔ HTTPS کار می‌کند${NC}"
else
    echo -e "${YELLOW}⚠️  HTTPS هنوز آماده نیست، چند دقیقه صبر کنید${NC}"
fi

# تست HTTP redirect
echo -e "${YELLOW}⏳ تست HTTP to HTTPS redirect...${NC}"
REDIRECT=$(curl -sI http://$DOMAIN | grep -i "location: https://" | wc -l)
if [ $REDIRECT -gt 0 ]; then
    echo -e "${GREEN}✔ HTTP به HTTPS redirect می‌شود${NC}"
else
    echo -e "${YELLOW}⚠️  Redirect هنوز فعال نشده${NC}"
fi

# نمایش وضعیت containers
echo ""
echo -e "${YELLOW}📊 وضعیت سرویس‌ها:${NC}"
sudo docker compose ps

################################################################################
# نمایش اطلاعات نهایی
################################################################################

clear
cat << "EOF"

════════════════════════════════════════════════════════════════
          ✅ SSL/HTTPS با موفقیت فعال شد! 🔒
════════════════════════════════════════════════════════════════

EOF

echo -e "${GREEN}🌐 دسترسی:${NC}"
echo -e "   HTTPS:  https://$DOMAIN"
echo -e "   HTTP:   http://$DOMAIN  (redirect به HTTPS)"
echo ""

echo -e "${GREEN}🔐 اطلاعات ورود:${NC}"
echo -e "   Email:    admin@mlm.com"
echo -e "   Password: admin123"
echo ""

echo -e "${GREEN}🔒 گواهی SSL:${NC}"
echo -e "   صادر کننده: Let's Encrypt"
echo -e "   اعتبار: 90 روز"
echo -e "   تمدید خودکار: ✅ فعال"
echo ""

echo -e "${GREEN}🔄 تمدید خودکار:${NC}"
echo -e "   Certbot هر 12 ساعت یکبار چک می‌کند"
echo -e "   30 روز قبل از انقضا، خودکار تمدید می‌شود"
echo -e "   Nginx بعد از تمدید خودکار reload می‌شود"
echo ""

echo -e "${YELLOW}📋 دستورات مفید:${NC}"
echo -e "   # چک کردن گواهی"
echo -e "   sudo certbot certificates"
echo ""
echo -e "   # تست تمدید دستی"
echo -e "   sudo certbot renew --dry-run"
echo ""
echo -e "   # تمدید دستی (اگر نیاز شد)"
echo -e "   sudo certbot renew"
echo ""
echo -e "   # مشاهده لاگ تمدید"
echo -e "   sudo journalctl -u certbot.timer"
echo ""

echo -e "${BLUE}⚠️  نکات مهم:${NC}"
echo -e "   1. گواهی هر 90 روز یکبار منقضی می‌شود"
echo -e "   2. تمدید خودکار 30 روز قبل شروع می‌شود"
echo -e "   3. ایمیل‌های هشدار به $EMAIL ارسال می‌شود"
echo -e "   4. در صورت مشکل، backup ها در همین پوشه هستند"
echo ""

echo "════════════════════════════════════════════════════════════════"
echo -e "${GREEN}✨ سیستم با HTTPS آماده است! ✨${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

# ذخیره اطلاعات SSL
cat > SSL_INFO.txt << INFO_EOF
MLM System - SSL Information
============================

Domain: $DOMAIN
Email: $EMAIL
Certificate Path: /etc/letsencrypt/live/$DOMAIN/
Issue Date: $(date)
Expiry Date: $(date -d "+90 days")

Auto-Renewal: ENABLED
Renewal Check: Every 12 hours
Renewal Trigger: 30 days before expiry

Access URLs:
- HTTPS: https://$DOMAIN
- API: https://$DOMAIN/api
- Health: https://$DOMAIN/health

Backup Files:
- nginx.conf.backup.*
- docker-compose.yml.backup.*
- frontend/.env.backup.*

Commands:
- Check certificate: sudo certbot certificates
- Test renewal: sudo certbot renew --dry-run
- Manual renewal: sudo certbot renew
- View logs: sudo journalctl -u certbot.timer

Generated: $(date)
INFO_EOF

echo -e "${GREEN}✔ اطلاعات SSL در فایل SSL_INFO.txt ذخیره شد${NC}"
echo ""
