#!/bin/bash

################################################################################
# MLM System - SSL Health Check & Troubleshooting
# چک کردن وضعیت SSL و رفع مشکلات
################################################################################

GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

clear
echo "════════════════════════════════════════════════════════════════"
echo -e "${BLUE}🔒 MLM System - SSL Health Check${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

# خواندن domain از SSL_INFO.txt یا nginx.conf
if [ -f "SSL_INFO.txt" ]; then
    DOMAIN=$(grep "Domain:" SSL_INFO.txt | awk '{print $2}')
elif [ -f "nginx.conf" ]; then
    DOMAIN=$(grep "server_name" nginx.conf | grep -v "_" | head -1 | awk '{print $2}' | tr -d ';')
else
    read -p "دامنه خود را وارد کنید: " DOMAIN
fi

if [ -z "$DOMAIN" ]; then
    echo -e "${RED}❌ دامنه یافت نشد${NC}"
    exit 1
fi

echo -e "${BLUE}🌐 دامنه: $DOMAIN${NC}"
echo ""

################################################################################
# 1. بررسی DNS
################################################################################

echo "════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}1️⃣  بررسی DNS...${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

DNS_IP=$(dig +short $DOMAIN | head -1)
SERVER_IP=$(curl -s ifconfig.me 2>/dev/null || hostname -I | awk '{print $1}')

echo -e "DNS IP:    $DNS_IP"
echo -e "Server IP: $SERVER_IP"
echo ""

if [ "$DNS_IP" = "$SERVER_IP" ]; then
    echo -e "${GREEN}✔ DNS به این سرور اشاره می‌کند${NC}"
else
    echo -e "${RED}✘ DNS به این سرور اشاره نمی‌کند!${NC}"
    echo -e "${YELLOW}⚠️  DNS را به $SERVER_IP تغییر دهید${NC}"
fi

################################################################################
# 2. بررسی پورت‌ها
################################################################################

echo ""
echo "════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}2️⃣  بررسی پورت‌ها...${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

# پورت 80
if sudo netstat -tlnp 2>/dev/null | grep -q ":80 " || sudo ss -tlnp 2>/dev/null | grep -q ":80 "; then
    echo -e "${GREEN}✔ پورت 80 در حال استفاده (OK)${NC}"
    PORT_80=$(sudo netstat -tlnp 2>/dev/null | grep ":80 " | awk '{print $7}' | head -1)
    echo -e "  استفاده توسط: $PORT_80"
else
    echo -e "${RED}✘ پورت 80 آزاد است (باید اشغال باشد)${NC}"
fi

echo ""

# پورت 443
if sudo netstat -tlnp 2>/dev/null | grep -q ":443 " || sudo ss -tlnp 2>/dev/null | grep -q ":443 "; then
    echo -e "${GREEN}✔ پورت 443 در حال استفاده (OK)${NC}"
    PORT_443=$(sudo netstat -tlnp 2>/dev/null | grep ":443 " | awk '{print $7}' | head -1)
    echo -e "  استفاده توسط: $PORT_443"
else
    echo -e "${RED}✘ پورت 443 آزاد است (SSL فعال نیست)${NC}"
fi

################################################################################
# 3. بررسی گواهی SSL
################################################################################

echo ""
echo "════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}3️⃣  بررسی گواهی SSL...${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

if [ -d "/etc/letsencrypt/live/$DOMAIN" ]; then
    echo -e "${GREEN}✔ گواهی SSL موجود است${NC}"
    echo ""
    
    # اطلاعات گواهی
    sudo certbot certificates -d $DOMAIN 2>/dev/null | grep -A 10 "Certificate Name: $DOMAIN"
    
    echo ""
    
    # تاریخ انقضا
    EXPIRY=$(sudo openssl x509 -enddate -noout -in /etc/letsencrypt/live/$DOMAIN/fullchain.pem 2>/dev/null | cut -d= -f2)
    if [ ! -z "$EXPIRY" ]; then
        EXPIRY_EPOCH=$(date -d "$EXPIRY" +%s)
        NOW_EPOCH=$(date +%s)
        DAYS_LEFT=$(( ($EXPIRY_EPOCH - $NOW_EPOCH) / 86400 ))
        
        echo -e "تاریخ انقضا: $EXPIRY"
        echo -e "روزهای باقی‌مانده: $DAYS_LEFT روز"
        
        if [ $DAYS_LEFT -lt 30 ]; then
            echo -e "${YELLOW}⚠️  کمتر از 30 روز به انقضا مانده! تمدید کنید.${NC}"
        else
            echo -e "${GREEN}✔ گواهی معتبر است${NC}"
        fi
    fi
else
    echo -e "${RED}✘ گواهی SSL یافت نشد${NC}"
    echo -e "${YELLOW}⚠️  ابتدا enable-ssl.sh را اجرا کنید${NC}"
fi

################################################################################
# 4. تست HTTPS Connection
################################################################################

echo ""
echo "════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}4️⃣  تست HTTPS Connection...${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

# تست health endpoint
echo -e "تست: https://$DOMAIN/health"
HTTPS_STATUS=$(curl -s -o /dev/null -w "%{http_code}" --max-time 10 https://$DOMAIN/health 2>/dev/null)

if [ "$HTTPS_STATUS" = "200" ]; then
    echo -e "${GREEN}✔ HTTPS کار می‌کند (Status: 200)${NC}"
elif [ "$HTTPS_STATUS" = "000" ]; then
    echo -e "${RED}✘ Connection failed (SSL/Network issue)${NC}"
else
    echo -e "${YELLOW}⚠️  Status: $HTTPS_STATUS${NC}"
fi

echo ""

# تست redirect از HTTP به HTTPS
echo -e "تست: HTTP to HTTPS Redirect"
HTTP_REDIRECT=$(curl -s -o /dev/null -w "%{http_code}" --max-time 10 http://$DOMAIN 2>/dev/null)

if [ "$HTTP_REDIRECT" = "301" ] || [ "$HTTP_REDIRECT" = "302" ]; then
    echo -e "${GREEN}✔ HTTP به HTTPS redirect می‌شود (Status: $HTTP_REDIRECT)${NC}"
else
    echo -e "${YELLOW}⚠️  Redirect Status: $HTTP_REDIRECT${NC}"
fi

################################################################################
# 5. بررسی Certbot Timer
################################################################################

echo ""
echo "════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}5️⃣  بررسی تمدید خودکار...${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

if sudo systemctl is-active --quiet certbot.timer; then
    echo -e "${GREEN}✔ Certbot timer فعال است${NC}"
    
    # زمان آخرین و بعدی اجرا
    echo ""
    sudo systemctl status certbot.timer | grep -A 2 "Trigger:"
else
    echo -e "${RED}✘ Certbot timer فعال نیست${NC}"
    echo -e "${YELLOW}⚠️  دستور: sudo systemctl enable --now certbot.timer${NC}"
fi

################################################################################
# 6. بررسی Docker Containers
################################################################################

echo ""
echo "════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}6️⃣  وضعیت Containers...${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

sudo docker compose ps

################################################################################
# 7. بررسی Nginx Logs
################################################################################

echo ""
echo "════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}7️⃣  آخرین خطاهای Nginx...${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

if sudo docker compose logs nginx --tail 20 2>/dev/null | grep -i error; then
    echo -e "${YELLOW}⚠️  خطاهایی در لاگ Nginx یافت شد${NC}"
else
    echo -e "${GREEN}✔ خطایی در لاگ Nginx یافت نشد${NC}"
fi

################################################################################
# 8. SSL Grade Test
################################################################################

echo ""
echo "════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}8️⃣  تست امنیت SSL...${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

echo -e "برای تست کامل امنیت SSL از:"
echo -e "${BLUE}https://www.ssllabs.com/ssltest/analyze.html?d=$DOMAIN${NC}"
echo ""

################################################################################
# خلاصه و توصیه‌ها
################################################################################

echo ""
echo "════════════════════════════════════════════════════════════════"
echo -e "${BLUE}📋 خلاصه و توصیه‌ها${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""

# شمارش مشکلات
ISSUES=0

if [ "$DNS_IP" != "$SERVER_IP" ]; then
    echo -e "${RED}❌ DNS به سرور اشاره ندارد${NC}"
    ISSUES=$((ISSUES + 1))
fi

if [ ! -d "/etc/letsencrypt/live/$DOMAIN" ]; then
    echo -e "${RED}❌ گواهی SSL یافت نشد${NC}"
    ISSUES=$((ISSUES + 1))
fi

if [ "$HTTPS_STATUS" != "200" ]; then
    echo -e "${RED}❌ HTTPS کار نمی‌کند${NC}"
    ISSUES=$((ISSUES + 1))
fi

if ! sudo systemctl is-active --quiet certbot.timer; then
    echo -e "${RED}❌ تمدید خودکار فعال نیست${NC}"
    ISSUES=$((ISSUES + 1))
fi

if [ $ISSUES -eq 0 ]; then
    echo -e "${GREEN}✅ همه چیز سالم است!${NC}"
else
    echo ""
    echo -e "${YELLOW}⚠️  $ISSUES مشکل یافت شد${NC}"
    echo ""
    echo "برای رفع مشکلات:"
    echo "1. اطمینان از تنظیم DNS"
    echo "2. اجرای: ./enable-ssl.sh"
    echo "3. چک کردن: sudo docker compose logs"
fi

echo ""
echo "════════════════════════════════════════════════════════════════"
echo -e "${BLUE}🔧 دستورات مفید:${NC}"
echo "════════════════════════════════════════════════════════════════"
echo ""
echo "# تمدید دستی گواهی"
echo "sudo certbot renew"
echo ""
echo "# تست تمدید"
echo "sudo certbot renew --dry-run"
echo ""
echo "# مشاهده لاگ Nginx"
echo "sudo docker compose logs -f nginx"
echo ""
echo "# Reload Nginx"
echo "sudo docker compose exec nginx nginx -s reload"
echo ""
echo "# Restart همه چیز"
echo "sudo docker compose restart"
echo ""
