import { useState, useEffect } from 'react';
import axios from 'axios';
import { Tabs, TabsList, TabsTrigger, TabsContent } from './ui/tabs';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from './ui/card';
import { Button } from './ui/button';
import { ExternalLink } from 'lucide-react';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

export default function LoginTabs() {
  const [tabs, setTabs] = useState([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchTabs();
  }, []);

  const fetchTabs = async () => {
    try {
      // Fetch public config
      const response = await axios.get(`${API}/user/config/public`);
      
      const tabsData = [];
      for (let i = 1; i <= 3; i++) {
        const key = `login_tab_${i}`;
        if (response.data[key]) {
          try {
            const tabData = JSON.parse(response.data[key]);
            // 🔧 بررسی enabled به صورت boolean و string
            const isEnabled = tabData.enabled === true || tabData.enabled === "true";
            if (isEnabled && tabData.title && tabData.url) {
              tabsData.push({
                number: i,
                ...tabData
              });
            }
          } catch (e) {
            console.error(`Error parsing tab ${i}:`, e);
          }
        }
      }
      
      setTabs(tabsData);
    } catch (error) {
      console.error('Error fetching tabs:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleOpenTab = (url) => {
    window.open(url, '_blank', 'noopener,noreferrer');
  };

  // اگر هیچ تبی فعال نیست، چیزی نمایش نده
  if (loading || tabs.length === 0) {
    return null;
  }

  return (
    <div className="w-full max-w-4xl mx-auto mt-8">
      <Tabs defaultValue={`tab-${tabs[0]?.number || 1}`} className="w-full">
        <TabsList className="grid w-full bg-white shadow-md rounded-xl p-1.5" style={{ gridTemplateColumns: `repeat(${tabs.length}, 1fr)` }}>
          {tabs.map(tab => (
            <TabsTrigger 
              key={tab.number} 
              value={`tab-${tab.number}`}
              className="text-sm font-bold rounded-lg py-3 data-[state=active]:bg-gradient-to-r data-[state=active]:from-blue-600 data-[state=active]:to-purple-600 data-[state=active]:text-white data-[state=active]:shadow-lg transition-all duration-200"
            >
              {tab.title}
            </TabsTrigger>
          ))}
        </TabsList>

        {tabs.map(tab => (
          <TabsContent key={tab.number} value={`tab-${tab.number}`} className="mt-6">
            <Card className="shadow-2xl border-0 bg-gradient-to-br from-white via-blue-50 to-purple-50 overflow-hidden">
              <CardHeader className="text-center space-y-4 pb-6 bg-gradient-to-r from-blue-600/10 to-purple-600/10">
                <div className="mx-auto w-20 h-20 bg-gradient-to-br from-blue-500 to-purple-600 rounded-full flex items-center justify-center shadow-xl animate-pulse">
                  <ExternalLink className="w-10 h-10 text-white" />
                </div>
                <CardTitle className="text-3xl font-bold bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
                  {tab.title}
                </CardTitle>
              </CardHeader>
              
              <CardContent className="space-y-6 pb-10">
                {tab.description && (
                  <div className="bg-white/80 backdrop-blur-sm border-r-4 border-blue-500 p-5 rounded-xl shadow-md">
                    <p className="text-gray-700 text-center text-base leading-relaxed font-medium">
                      {tab.description}
                    </p>
                  </div>
                )}

                <div className="flex justify-center pt-4">
                  <Button
                    onClick={() => handleOpenTab(tab.url)}
                    size="lg"
                    className="bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white font-bold px-10 py-7 text-lg shadow-xl hover:shadow-2xl transition-all duration-300 hover:scale-110 transform rounded-xl"
                  >
                    <ExternalLink className="w-6 h-6 ml-2" />
                    {tab.button_text || `مشاهده ${tab.title}`}
                  </Button>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        ))}
      </Tabs>
    </div>
  );
}
