import { useState, useEffect } from 'react';
import axios from 'axios';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from './ui/dialog';
import { Button } from './ui/button';
import { X } from 'lucide-react';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

export default function AnnouncementModal() {
  const [open, setOpen] = useState(false);
  const [announcement, setAnnouncement] = useState(null);

  useEffect(() => {
    checkAnnouncement();
  }, []);

  const checkAnnouncement = async () => {
    try {
      const token = localStorage.getItem('token');
      if (!token) return;

      const res = await axios.get(`${API}/user/announcements/modal`, {
        headers: { Authorization: `Bearer ${token}` }
      });

      if (res.data.has_announcement) {
        setAnnouncement(res.data.announcement);
        setOpen(true);
      }
    } catch (error) {
      console.error('Error checking announcement:', error);
    }
  };

  const handleClose = async () => {
    if (announcement) {
      try {
        const token = localStorage.getItem('token');
        await axios.post(
          `${API}/user/announcements/${announcement.announcement_id}/mark-seen`,
          {},
          { headers: { Authorization: `Bearer ${token}` } }
        );
      } catch (error) {
        console.error('Error marking as seen:', error);
      }
    }
    setOpen(false);
  };

  // تابع برای اطمینان از صحت فرمت لینک
  const normalizeUrl = (url) => {
    if (!url) return '';
    // اگر قبلاً هر نوع پروتکلی دارد (http://, https://, ftp://, وغیره)، دست نزن
    if (url.includes('://')) {
      return url;
    }
    // در غیر این صورت، https:// اضافه کن
    return `https://${url}`;
  };

  if (!announcement) return null;

  return (
    <Dialog open={open} onOpenChange={handleClose}>
      <DialogContent className="sm:max-w-md">
        <DialogHeader>
          <DialogTitle className="text-xl font-bold text-blue-900 flex items-center justify-between">
            <span>📢 {announcement.title}</span>
            <button onClick={handleClose} className="text-gray-500 hover:text-gray-700">
              <X className="w-5 h-5" />
            </button>
          </DialogTitle>
        </DialogHeader>
        <div className="py-4">
          <p className="text-gray-700 whitespace-pre-wrap">{announcement.content}</p>
          {announcement.link && (
            <a
              href={normalizeUrl(announcement.link)}
              target="_blank"
              rel="noopener noreferrer"
              className="mt-4 inline-block text-blue-600 underline hover:text-blue-800"
            >
              مشاهده جزئیات بیشتر
            </a>
          )}
        </div>
        <Button onClick={handleClose} className="w-full">
          متوجه شدم
        </Button>
      </DialogContent>
    </Dialog>
  );
}
