import { useState, useEffect } from 'react';
import axios from 'axios';
import { toast } from 'sonner';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Textarea } from './ui/textarea';
import { Switch } from './ui/switch';
import { Trash2, Plus, Bell, Megaphone } from 'lucide-react';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

export default function AdminAnnouncements() {
  const [announcements, setAnnouncements] = useState([]);
  const [title, setTitle] = useState('');
  const [content, setContent] = useState('');
  const [link, setLink] = useState('');
  const [isModal, setIsModal] = useState(false);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    fetchAnnouncements();
  }, []);

  const fetchAnnouncements = async () => {
    try {
      const token = localStorage.getItem('token');
      const res = await axios.get(`${API}/admin/announcements`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setAnnouncements(res.data);
    } catch (error) {
      toast.error('خطا در دریافت اطلاعیه‌ها');
    }
  };

  const handleCreate = async (e) => {
    e.preventDefault();
    
    if (!title || !content) {
      toast.error('عنوان و متن اجباری است');
      return;
    }

    setLoading(true);
    try {
      const token = localStorage.getItem('token');
      await axios.post(
        `${API}/admin/announcements`,
        { title, content, link: link || null, is_modal: isModal },
        { headers: { Authorization: `Bearer ${token}` } }
      );

      toast.success('✅ اطلاعیه ایجاد شد');
      setTitle('');
      setContent('');
      setLink('');
      setIsModal(false);
      fetchAnnouncements();
    } catch (error) {
      toast.error('خطا در ایجاد اطلاعیه');
    } finally {
      setLoading(false);
    }
  };

  const handleDelete = async (id) => {
    if (!confirm('آیا از حذف این اطلاعیه اطمینان دارید؟')) return;

    try {
      const token = localStorage.getItem('token');
      await axios.delete(`${API}/admin/announcements/${id}`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      toast.success('✅ اطلاعیه حذف شد');
      fetchAnnouncements();
    } catch (error) {
      toast.error('خطا در حذف اطلاعیه');
    }
  };

  return (
    <div className="space-y-6">
      <Card>
        <CardHeader className="bg-gradient-to-r from-purple-500 to-pink-600 text-white">
          <CardTitle className="flex items-center gap-2">
            <Plus className="w-5 h-5" />
            ایجاد اطلاعیه جدید
          </CardTitle>
        </CardHeader>
        <CardContent className="p-6">
          <form onSubmit={handleCreate} className="space-y-4">
            <div>
              <Label htmlFor="title">عنوان</Label>
              <Input
                id="title"
                value={title}
                onChange={(e) => setTitle(e.target.value)}
                placeholder="عنوان اطلاعیه"
              />
            </div>

            <div>
              <Label htmlFor="content">متن</Label>
              <Textarea
                id="content"
                value={content}
                onChange={(e) => setContent(e.target.value)}
                placeholder="متن کامل اطلاعیه..."
                rows={4}
              />
            </div>

            <div>
              <Label htmlFor="link">لینک (اختیاری)</Label>
              <Input
                id="link"
                value={link}
                onChange={(e) => setLink(e.target.value)}
                placeholder="https://example.com"
              />
              <p className="text-xs text-gray-500 mt-1">
                💡 اگر لینک شما https:// ندارد، به صورت خودکار اضافه می‌شود
              </p>
            </div>

            <div className="flex items-center justify-between p-4 bg-blue-50 rounded-lg">
              <div>
                <Label htmlFor="is-modal" className="font-semibold">
                  {isModal ? '🔔 اعلان مودال (یکبار نمایش)' : '📢 اسلایدر (نمایش مداوم)'}
                </Label>
                <p className="text-sm text-gray-600 mt-1">
                  {isModal 
                    ? 'این پیام یکبار به صورت popup برای همه نمایش داده می‌شود'
                    : 'این پیام در اسلایدر بالای داشبورد نمایش داده می‌شود'
                  }
                </p>
              </div>
              <Switch
                id="is-modal"
                checked={isModal}
                onCheckedChange={setIsModal}
              />
            </div>

            <Button
              type="submit"
              disabled={loading}
              className="w-full bg-gradient-to-r from-purple-600 to-pink-600"
            >
              {loading ? 'در حال ایجاد...' : 'ایجاد اطلاعیه'}
            </Button>
          </form>
        </CardContent>
      </Card>

      <Card>
        <CardHeader>
          <CardTitle>لیست اطلاعیه‌ها ({announcements.length})</CardTitle>
        </CardHeader>
        <CardContent>
          {announcements.length === 0 ? (
            <p className="text-center text-gray-500 py-8">هیچ اطلاعیه‌ای وجود ندارد</p>
          ) : (
            <div className="space-y-3">
              {announcements.map((ann) => (
                <div
                  key={ann.announcement_id}
                  className="border-2 rounded-lg p-4 flex items-start justify-between"
                >
                  <div className="flex-1">
                    <div className="flex items-center gap-2 mb-2">
                      {ann.is_modal ? (
                        <Bell className="w-4 h-4 text-blue-600" />
                      ) : (
                        <Megaphone className="w-4 h-4 text-green-600" />
                      )}
                      <h3 className="font-semibold text-lg">{ann.title}</h3>
                      <span className={`text-xs px-2 py-1 rounded ${
                        ann.is_modal ? 'bg-blue-100 text-blue-700' : 'bg-green-100 text-green-700'
                      }`}>
                        {ann.is_modal ? 'مودال' : 'اسلایدر'}
                      </span>
                    </div>
                    <p className="text-gray-700 text-sm mb-2">{ann.content}</p>
                    {ann.link && (
                      <a
                        href={ann.link}
                        target="_blank"
                        rel="noopener noreferrer"
                        className="text-xs text-blue-600 hover:underline"
                      >
                        {ann.link}
                      </a>
                    )}
                    <p className="text-xs text-gray-400 mt-2">
                      {new Date(ann.created_at).toLocaleDateString('fa-IR')}
                    </p>
                  </div>
                  <Button
                    variant="destructive"
                    size="sm"
                    onClick={() => handleDelete(ann.announcement_id)}
                  >
                    <Trash2 className="w-4 h-4" />
                  </Button>
                </div>
              ))}
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
